/*
 * Decompiled with CFR 0.152.
 */
package net.opentsdb.tools;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import net.opentsdb.core.TSDB;
import net.opentsdb.meta.TSMeta;
import net.opentsdb.tools.ArgP;
import net.opentsdb.tools.CliOptions;
import net.opentsdb.tools.CliUtils;
import net.opentsdb.tools.MetaPurge;
import net.opentsdb.tools.MetaSync;
import net.opentsdb.tools.TreeSync;
import net.opentsdb.uid.NoSuchUniqueId;
import net.opentsdb.uid.NoSuchUniqueName;
import net.opentsdb.uid.UniqueId;
import net.opentsdb.utils.Config;
import org.hbase.async.AtomicIncrementRequest;
import org.hbase.async.Bytes;
import org.hbase.async.DeleteRequest;
import org.hbase.async.GetRequest;
import org.hbase.async.HBaseClient;
import org.hbase.async.HBaseException;
import org.hbase.async.KeyValue;
import org.hbase.async.PutRequest;
import org.hbase.async.Scanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class UidManager {
    private static final Logger LOG = LoggerFactory.getLogger(UidManager.class);

    UidManager() {
    }

    static void usage(String errmsg) {
        UidManager.usage(null, errmsg);
    }

    static void usage(ArgP argp, String errmsg) {
        System.err.println(errmsg);
        System.err.println("Usage: uid <subcommand> args\nSub commands:\n  grep [kind] <RE>: Finds matching IDs.\n  assign <kind> <name> [names]: Assign an ID for the given name(s).\n  rename <kind> <name> <newname>: Renames this UID.\n  delete <kind> <name>: Deletes this UID.\n  fsck: [fix] [delete_unknown] Checks the consistency of UIDs.\n        fix            - Fix errors. By default errors are logged.\n        delete_unknown - Remove columns with unknown qualifiers.\n                         The \"fix\" flag must be supplied as well.\n\n  [kind] <name>: Lookup the ID of this name.\n  [kind] <ID>: Lookup the name of this ID.\n  metasync: Generates missing TSUID and UID meta entries, updates\n            created timestamps\n  metapurge: Removes meta data entries from the UID table\n  treesync: Process all timeseries meta objects through tree rules\n  treepurge <id> [definition]: Purge a tree and/or the branches\n            from storage. Provide an integer Tree ID and optionally\n            add \"true\" to delete the tree definition\n\nExample values for [kind]: metrics, tagk (tag name), tagv (tag value).");
        if (argp != null) {
            System.err.print(argp.usage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws Exception {
        int rc;
        short idwidth;
        ArgP argp = new ArgP();
        CliOptions.addCommon(argp);
        CliOptions.addVerbose(argp);
        argp.addOption("--idwidth", "N", "Number of bytes on which the UniqueId is encoded.");
        argp.addOption("--ignore-case", "Ignore case distinctions when matching a regexp.");
        argp.addOption("-i", "Short for --ignore-case.");
        args = CliOptions.parse(argp, args);
        if (args == null) {
            UidManager.usage(argp, "Invalid usage");
            System.exit(2);
        } else if (args.length < 1) {
            UidManager.usage(argp, "Not enough arguments");
            System.exit(2);
        }
        short s = idwidth = argp.has("--idwidth") ? (short)Short.parseShort(argp.get("--idwidth")) : (short)3;
        if (idwidth <= 0) {
            UidManager.usage(argp, "Negative or 0 --idwidth");
            System.exit(3);
        }
        boolean ignorecase = argp.has("--ignore-case") || argp.has("-i");
        Config config = CliOptions.getConfig(argp);
        byte[] table = config.getString("tsd.storage.hbase.uid_table").getBytes();
        TSDB tsdb = new TSDB(config);
        tsdb.getClient().ensureTableExists(config.getString("tsd.storage.hbase.uid_table")).joinUninterruptibly();
        argp = null;
        try {
            rc = UidManager.runCommand(tsdb, table, idwidth, ignorecase, args);
        }
        finally {
            try {
                LOG.info("Shutting down TSD....");
                tsdb.getClient().shutdown().joinUninterruptibly();
                LOG.info("Gracefully shutdown the TSD");
            }
            catch (Exception e) {
                LOG.error("Unexpected exception while shutting down", (Throwable)e);
                rc = 42;
            }
        }
        System.exit(rc);
    }

    private static int runCommand(TSDB tsdb, byte[] table, short idwidth, boolean ignorecase, String[] args) {
        int nargs = args.length;
        if (args[0].equals("grep")) {
            if (2 <= nargs && nargs <= 3) {
                try {
                    return UidManager.grep(tsdb.getClient(), table, ignorecase, args);
                }
                catch (HBaseException e) {
                    return 3;
                }
            }
            UidManager.usage("Wrong number of arguments");
            return 2;
        }
        if (args[0].equals("assign")) {
            if (nargs < 3) {
                UidManager.usage("Wrong number of arguments");
                return 2;
            }
            return UidManager.assign(tsdb, table, idwidth, args);
        }
        if (args[0].equals("rename")) {
            if (nargs != 4) {
                UidManager.usage("Wrong number of arguments");
                return 2;
            }
            return UidManager.rename(tsdb.getClient(), table, idwidth, args);
        }
        if (args[0].equals("delete")) {
            if (nargs != 3) {
                UidManager.usage("Wrong number of arguments");
                return 2;
            }
            try {
                return UidManager.delete(tsdb, table, args);
            }
            catch (Exception e) {
                LOG.error("Unexpected exception", (Throwable)e);
                return 4;
            }
        }
        if (args[0].equals("fsck")) {
            boolean fix = false;
            boolean fix_unknowns = false;
            if (args.length > 1) {
                for (String arg : args) {
                    if (arg.equals("fix")) {
                        fix = true;
                        continue;
                    }
                    if (!arg.equals("delete_unknown")) continue;
                    fix_unknowns = true;
                }
            }
            return UidManager.fsck(tsdb.getClient(), table, fix, fix_unknowns);
        }
        if (args[0].equals("metasync")) {
            try {
                tsdb.getClient().ensureTableExists(tsdb.getConfig().getString("tsd.storage.hbase.data_table")).joinUninterruptibly();
                tsdb.initializePlugins(false);
                return UidManager.metaSync(tsdb);
            }
            catch (Exception e) {
                LOG.error("Unexpected exception", (Throwable)e);
                return 3;
            }
        }
        if (args[0].equals("metapurge")) {
            try {
                tsdb.getClient().ensureTableExists(tsdb.getConfig().getString("tsd.storage.hbase.uid_table")).joinUninterruptibly();
                return UidManager.metaPurge(tsdb);
            }
            catch (Exception e) {
                LOG.error("Unexpected exception", (Throwable)e);
                return 3;
            }
        }
        if (args[0].equals("treesync")) {
            try {
                tsdb.getClient().ensureTableExists(tsdb.getConfig().getString("tsd.storage.hbase.uid_table")).joinUninterruptibly();
                if (!tsdb.getConfig().enable_tree_processing()) {
                    LOG.warn("Tree processing is disabled");
                    return 0;
                }
                return UidManager.treeSync(tsdb);
            }
            catch (Exception e) {
                LOG.error("Unexpected exception", (Throwable)e);
                return 3;
            }
        }
        if (args[0].equals("treepurge")) {
            if (nargs < 2) {
                UidManager.usage("Wrong number of arguments");
                return 2;
            }
            try {
                String delete_all;
                tsdb.getClient().ensureTableExists(tsdb.getConfig().getString("tsd.storage.hbase.uid_table")).joinUninterruptibly();
                int tree_id = Integer.parseInt(args[1]);
                boolean delete_definitions = nargs < 3 ? false : (delete_all = args[2]).toLowerCase().equals("true");
                return UidManager.purgeTree(tsdb, tree_id, delete_definitions);
            }
            catch (Exception e) {
                LOG.error("Unexpected exception", (Throwable)e);
                return 3;
            }
        }
        if (1 <= nargs && nargs <= 2) {
            String kind = nargs == 2 ? args[0] : null;
            try {
                long id = Long.parseLong(args[nargs - 1]);
                return UidManager.lookupId(tsdb.getClient(), table, idwidth, id, kind);
            }
            catch (NumberFormatException e) {
                return UidManager.lookupName(tsdb.getClient(), table, idwidth, args[nargs - 1], kind);
            }
        }
        UidManager.usage("Wrong number of arguments");
        return 2;
    }

    private static int grep(HBaseClient client, byte[] table, boolean ignorecase, String[] args) {
        String regexp;
        Scanner scanner = client.newScanner(table);
        scanner.setMaxNumRows(1024);
        scanner.setFamily(CliUtils.ID_FAMILY);
        if (args.length == 3) {
            scanner.setQualifier(CliUtils.toBytes(args[1]));
            regexp = args[2];
        } else {
            regexp = args[1];
        }
        if (ignorecase) {
            regexp = "(?i)" + regexp;
        }
        scanner.setKeyRegexp(regexp, CliUtils.CHARSET);
        boolean found = false;
        try {
            ArrayList rows;
            while ((rows = (ArrayList)scanner.nextRows().joinUninterruptibly()) != null) {
                for (ArrayList row : rows) {
                    found |= UidManager.printResult(row, CliUtils.ID_FAMILY, true);
                }
            }
        }
        catch (HBaseException e) {
            LOG.error("Error while scanning HBase, scanner=" + scanner, (Throwable)e);
            throw e;
        }
        catch (Exception e) {
            LOG.error("WTF?  Unexpected exception type, scanner=" + scanner, (Throwable)e);
            throw new AssertionError((Object)"Should never happen");
        }
        return found ? 0 : 1;
    }

    private static boolean printResult(ArrayList<KeyValue> row, byte[] family, boolean formard) {
        if (null == row || row.isEmpty()) {
            return false;
        }
        byte[] key = row.get(0).key();
        String name = formard ? CliUtils.fromBytes(key) : null;
        String id = formard ? null : Arrays.toString(key);
        boolean printed = false;
        for (KeyValue kv : row) {
            if (!Bytes.equals((byte[])kv.family(), (byte[])family)) continue;
            printed = true;
            if (formard) {
                id = Arrays.toString(kv.value());
            } else {
                name = CliUtils.fromBytes(kv.value());
            }
            System.out.println(CliUtils.fromBytes(kv.qualifier()) + ' ' + name + ": " + id);
        }
        return printed;
    }

    private static int assign(TSDB tsdb, byte[] table, short idwidth, String[] args) {
        boolean randomize = false;
        if (UniqueId.stringToUniqueIdType(args[1]) == UniqueId.UniqueIdType.METRIC) {
            randomize = tsdb.getConfig().getBoolean("tsd.core.uid.random_metrics");
        }
        UniqueId uid = new UniqueId(tsdb.getClient(), table, args[1], (int)idwidth, randomize);
        for (int i = 2; i < args.length; ++i) {
            try {
                uid.getOrCreateId(args[i]);
                UidManager.extactLookupName(tsdb.getClient(), table, idwidth, args[1], args[i]);
                continue;
            }
            catch (HBaseException e) {
                LOG.error("error while processing " + args[i], (Throwable)e);
                return 3;
            }
        }
        return 0;
    }

    private static int rename(HBaseClient client, byte[] table, short idwidth, String[] args) {
        String kind = args[1];
        String oldname = args[2];
        String newname = args[3];
        UniqueId uid = new UniqueId(client, table, kind, idwidth);
        try {
            uid.rename(oldname, newname);
        }
        catch (HBaseException e) {
            LOG.error("error while processing renaming " + oldname + " to " + newname, (Throwable)e);
            return 3;
        }
        catch (NoSuchUniqueName e) {
            LOG.error(e.getMessage());
            return 1;
        }
        System.out.println(kind + ' ' + oldname + " -> " + newname);
        return 0;
    }

    private static int delete(TSDB tsdb, byte[] table, String[] args) throws Exception {
        String kind = args[1];
        String name = args[2];
        try {
            tsdb.deleteUidAsync(kind, name).join();
        }
        catch (HBaseException e) {
            LOG.error("error while processing delete " + name, (Throwable)e);
            return 3;
        }
        catch (NoSuchUniqueName e) {
            LOG.error(e.getMessage());
            return 1;
        }
        LOG.info("UID " + kind + ' ' + name + " deleted.");
        return 0;
    }

    /*
     * WARNING - void declaration
     */
    private static int fsck(final HBaseClient client, final byte[] table, boolean fix, boolean fix_unknowns) {
        if (fix) {
            LOG.info("----------------------------------");
            LOG.info("-    Running fsck in FIX mode    -");
            LOG.info("-      Remove Unknowns: " + fix_unknowns + "     -");
            LOG.info("----------------------------------");
        } else {
            LOG.info("Running in log only mode");
        }
        long start_time = System.nanoTime();
        final class Uids {
            int errors;
            long maxid;
            long max_found_id;
            short width;
            final HashMap<String, String> id2name = new HashMap();
            final HashMap<String, String> name2id = new HashMap();

            Uids() {
            }

            void error(KeyValue kv, String msg) {
                this.error(msg + ".  kv=" + kv);
            }

            void error(String msg) {
                LOG.error(msg);
                ++this.errors;
            }

            void restoreReverseMap(String kind, String name, String uid) {
                PutRequest put = new PutRequest(table, UniqueId.stringToUid(uid), CliUtils.NAME_FAMILY, CliUtils.toBytes(kind), CliUtils.toBytes(name));
                client.put(put);
                this.id2name.put(uid, name);
                LOG.info("FIX: Restoring " + kind + " reverse mapping: " + uid + " -> " + name);
            }

            void removeReverseMap(String kind, String name, String uid) {
                byte[][] qualifiers = new byte[2][];
                qualifiers[0] = CliUtils.toBytes(kind);
                if (Bytes.equals((byte[])CliUtils.METRICS, (byte[])qualifiers[0])) {
                    qualifiers[1] = CliUtils.METRICS_META;
                } else if (Bytes.equals((byte[])CliUtils.TAGK, (byte[])qualifiers[0])) {
                    qualifiers[1] = CliUtils.TAGK_META;
                } else if (Bytes.equals((byte[])CliUtils.TAGV, (byte[])qualifiers[0])) {
                    qualifiers[1] = CliUtils.TAGV_META;
                }
                DeleteRequest delete = new DeleteRequest(table, UniqueId.stringToUid(uid), CliUtils.NAME_FAMILY, (byte[][])qualifiers);
                client.delete(delete);
                LOG.info("FIX: Removed " + kind + " reverse mapping: " + uid + " -> " + name);
            }
        }
        HashMap<String, Uids> name2uids = new HashMap<String, Uids>();
        Scanner scanner = client.newScanner(table);
        scanner.setMaxNumRows(1024);
        int kvcount = 0;
        try {
            ArrayList rows;
            while ((rows = (ArrayList)scanner.nextRows().joinUninterruptibly()) != null) {
                for (ArrayList arrayList : rows) {
                    for (KeyValue kv : arrayList) {
                        Object skey;
                        void var16_24;
                        ++kvcount;
                        byte[] qualifier = kv.qualifier();
                        if (Bytes.equals((byte[])qualifier, (byte[])TSMeta.META_QUALIFIER()) || Bytes.equals((byte[])qualifier, (byte[])TSMeta.COUNTER_QUALIFIER()) || Bytes.equals((byte[])qualifier, (byte[])CliUtils.METRICS_META) || Bytes.equals((byte[])qualifier, (byte[])CliUtils.TAGK_META) || Bytes.equals((byte[])qualifier, (byte[])CliUtils.TAGV_META)) continue;
                        if (!(Bytes.equals((byte[])qualifier, (byte[])CliUtils.METRICS) || Bytes.equals((byte[])qualifier, (byte[])CliUtils.TAGK) || Bytes.equals((byte[])qualifier, (byte[])CliUtils.TAGV))) {
                            LOG.warn("Unknown qualifier " + UniqueId.uidToString(qualifier) + " in row " + UniqueId.uidToString(kv.key()));
                            if (!fix || !fix_unknowns) continue;
                            DeleteRequest delete = new DeleteRequest(table, kv.key(), kv.family(), qualifier);
                            client.delete(delete);
                            LOG.info("FIX: Removed unknown qualifier " + UniqueId.uidToString(qualifier) + " in row " + UniqueId.uidToString(kv.key()));
                            continue;
                        }
                        String kind = CliUtils.fromBytes(kv.qualifier());
                        Uids uids = (Uids)name2uids.get(kind);
                        if (uids == null) {
                            Uids uids2 = new Uids();
                            name2uids.put(kind, uids2);
                        }
                        byte[] key = kv.key();
                        byte[] family = kv.family();
                        byte[] value = kv.value();
                        if (Bytes.equals((byte[])key, (byte[])CliUtils.MAXID_ROW)) {
                            if (value.length != 8) {
                                var16_24.error(kv, "Invalid maximum ID for " + kind + ": should be on 8 bytes: ");
                                continue;
                            }
                            var16_24.maxid = Bytes.getLong((byte[])value);
                            LOG.info("Maximum ID for " + kind + ": " + var16_24.maxid);
                            continue;
                        }
                        short idwidth = 0;
                        if (Bytes.equals((byte[])family, (byte[])CliUtils.ID_FAMILY)) {
                            String id;
                            idwidth = (short)value.length;
                            skey = CliUtils.fromBytes(key);
                            String string = UniqueId.uidToString(value);
                            long max_found_id = Bytes.equals((byte[])qualifier, (byte[])CliUtils.METRICS) ? UniqueId.uidToLong(value, TSDB.metrics_width()) : (Bytes.equals((byte[])qualifier, (byte[])CliUtils.TAGK) ? UniqueId.uidToLong(value, TSDB.tagk_width()) : UniqueId.uidToLong(value, TSDB.tagv_width()));
                            if (var16_24.max_found_id < max_found_id) {
                                var16_24.max_found_id = max_found_id;
                            }
                            if ((id = var16_24.name2id.put((String)skey, string)) != null) {
                                var16_24.error(kv, "Duplicate forward " + kind + " mapping: " + (String)skey + " -> " + id + " and " + (String)skey + " -> " + string);
                            }
                        } else if (Bytes.equals((byte[])family, (byte[])CliUtils.NAME_FAMILY)) {
                            skey = UniqueId.uidToString(key);
                            String string = CliUtils.fromBytes(value);
                            idwidth = (short)key.length;
                            String name = var16_24.id2name.put((String)skey, string);
                            if (name != null) {
                                var16_24.error(kv, "Duplicate reverse " + kind + "  mapping: " + string + " -> " + name + " and " + string + " -> " + (String)skey);
                            }
                        }
                        if (var16_24.width == 0) {
                            var16_24.width = idwidth;
                            continue;
                        }
                        if (var16_24.width == idwidth) continue;
                        var16_24.error(kv, "Invalid " + kind + " ID of length " + idwidth + " (expected: " + var16_24.width + ')');
                    }
                }
            }
        }
        catch (HBaseException e) {
            LOG.error("Error while scanning HBase, scanner=" + scanner, (Throwable)e);
            throw e;
        }
        catch (Exception e) {
            LOG.error("WTF?  Unexpected exception type, scanner=" + scanner, (Throwable)e);
            throw new AssertionError((Object)"Should never happen");
        }
        int errors = 0;
        for (Map.Entry entry : name2uids.entrySet()) {
            Object found;
            String id;
            String name;
            String kind = (String)entry.getKey();
            Uids uids = (Uids)entry.getValue();
            HashMap uid_collisions = null;
            for (Map.Entry<String, String> entry2 : uids.name2id.entrySet()) {
                name = entry2.getKey();
                id = entry2.getValue();
                found = uids.id2name.get(id);
                if (found == null) {
                    uids.error("Forward " + kind + " mapping is missing reverse mapping: " + name + " -> " + id);
                    if (!fix) continue;
                    uids.restoreReverseMap(kind, name, id);
                    continue;
                }
                if (((String)found).equals(name)) continue;
                uids.error("Forward " + kind + " mapping " + name + " -> " + id + " is different than reverse mapping: " + id + " -> " + (String)found);
                String id2 = uids.name2id.get(found);
                if (id2 != null) {
                    uids.error("Inconsistent forward " + kind + " mapping " + name + " -> " + id + " vs " + name + " -> " + (String)found + " / " + (String)found + " -> " + id2);
                    if (!fix) continue;
                    if (uid_collisions == null) {
                        uid_collisions = new HashMap(uids.name2id.size());
                        for (Map.Entry entry3 : uids.name2id.entrySet()) {
                            TreeSet names = (TreeSet)uid_collisions.get(entry3.getValue());
                            if (names == null) {
                                names = new TreeSet();
                                uid_collisions.put(entry3.getValue(), names);
                            }
                            names.add(entry3.getKey());
                        }
                    }
                    if (!uid_collisions.containsKey(id) || ((TreeSet)uid_collisions.get(id)).size() > 1) continue;
                    uids.restoreReverseMap(kind, name, id);
                    continue;
                }
                uids.error("Duplicate forward " + kind + " mapping " + name + " -> " + id + " and " + id2 + " -> " + (String)found);
                if (!fix) continue;
                uids.restoreReverseMap(kind, name, id);
            }
            if (uid_collisions != null) {
                for (Map.Entry<String, String> entry4 : uid_collisions.entrySet()) {
                    if (((TreeSet)((Object)entry4.getValue())).size() <= 1) continue;
                    StringBuilder fsck_builder = new StringBuilder("fsck");
                    id = entry4.getKey();
                    for (String name2 : (TreeSet)((Object)entry4.getValue())) {
                        fsck_builder.append(".").append(name2);
                        DeleteRequest delete = new DeleteRequest(table, CliUtils.toBytes(name2), CliUtils.ID_FAMILY, CliUtils.toBytes(kind));
                        client.delete(delete);
                        uids.name2id.remove(name2);
                        LOG.info("FIX: Removed forward " + kind + " mapping for " + name2 + " -> " + id);
                    }
                    String fsck_name = fsck_builder.toString();
                    PutRequest put = new PutRequest(table, CliUtils.toBytes(fsck_name), CliUtils.ID_FAMILY, CliUtils.toBytes(kind), UniqueId.stringToUid(id));
                    client.put(put);
                    LOG.info("FIX: Created forward " + kind + " mapping for fsck'd UID " + fsck_name + " -> " + entry4.getKey());
                    uids.name2id.put(fsck_name, entry4.getKey());
                    uids.restoreReverseMap(kind, fsck_name, id);
                    LOG.error("----------------------------------");
                    LOG.error("-     UID COLLISION DETECTED     -");
                    LOG.error("Corrupted UID [" + entry4.getKey() + "] renamed to [" + fsck_name + "]");
                    LOG.error("----------------------------------");
                }
            }
            for (Map.Entry<String, String> entry5 : uids.id2name.entrySet()) {
                name = entry5.getValue();
                id = entry5.getKey();
                found = uids.name2id.get(name);
                if (found == null) {
                    LOG.warn("Reverse " + kind + " mapping is missing forward mapping: " + name + " -> " + id);
                    if (!fix) continue;
                    uids.removeReverseMap(kind, name, id);
                    continue;
                }
                if (((String)found).equals(id)) continue;
                String name2 = uids.id2name.get(found);
                if (name2 != null) {
                    uids.error("Inconsistent reverse " + kind + " mapping " + id + " -> " + name + " vs " + (String)found + " -> " + name + " / " + name2 + " -> " + (String)found);
                    if (!fix) continue;
                    uids.removeReverseMap(kind, name, id);
                    continue;
                }
                uids.error("Duplicate reverse " + kind + " mapping " + id + " -> " + name + " and " + (String)found + " -> " + name2);
                if (!fix) continue;
                uids.removeReverseMap(kind, name, id);
            }
            int maxsize = Math.max(uids.id2name.size(), uids.name2id.size());
            if (uids.maxid > (long)maxsize) {
                LOG.warn("Max ID for " + kind + " is " + uids.maxid + " but only " + maxsize + " entries were found.  Maybe " + (uids.maxid - (long)maxsize) + " IDs were deleted?");
            } else if (uids.maxid < uids.max_found_id) {
                uids.error("We found an ID of " + uids.max_found_id + " for " + kind + " but the max ID is only " + uids.maxid + "!  Future IDs may be double-assigned!");
                if (fix) {
                    if (uids.max_found_id == Long.MAX_VALUE) {
                        LOG.error("Ran out of UIDs for " + kind + ". Unable to fix max ID");
                    } else {
                        long l = uids.max_found_id - uids.maxid;
                        AtomicIncrementRequest air = new AtomicIncrementRequest(table, CliUtils.MAXID_ROW, CliUtils.ID_FAMILY, CliUtils.toBytes(kind), l);
                        client.atomicIncrement(air);
                        LOG.info("FIX: Updated max ID for " + kind + " to " + uids.max_found_id);
                    }
                }
            }
            if (uids.errors <= 0) continue;
            LOG.error(kind + ": Found " + uids.errors + " errors.");
            errors += uids.errors;
        }
        long timing = (System.nanoTime() - start_time) / 1000000L;
        LOG.info(kvcount + " KVs analyzed in " + timing + "ms (~" + (long)(kvcount * 1000) / timing + " KV/s)");
        if (errors == 0) {
            LOG.info("No errors found.");
            return 0;
        }
        LOG.warn(errors + " errors found.");
        return errors;
    }

    private static int lookupId(HBaseClient client, byte[] table, short idwidth, long lid, String kind) {
        byte[] id = UidManager.idInBytes(idwidth, lid);
        if (id == null) {
            return 1;
        }
        if (kind != null) {
            return UidManager.extactLookupId(client, table, idwidth, kind, id);
        }
        return UidManager.findAndPrintRow(client, table, id, CliUtils.NAME_FAMILY, false);
    }

    private static int findAndPrintRow(HBaseClient client, byte[] table, byte[] key, byte[] family, boolean formard) {
        ArrayList row;
        GetRequest get = new GetRequest(table, key);
        get.family(family);
        try {
            row = (ArrayList)client.get(get).joinUninterruptibly();
        }
        catch (HBaseException e) {
            LOG.error("Get failed: " + get, (Throwable)e);
            return 1;
        }
        catch (Exception e) {
            LOG.error("WTF?  Unexpected exception type, get=" + get, (Throwable)e);
            return 42;
        }
        return UidManager.printResult(row, family, formard) ? 0 : 1;
    }

    private static int extactLookupId(HBaseClient client, byte[] table, short idwidth, String kind, byte[] id) {
        UniqueId uid = new UniqueId(client, table, kind, idwidth);
        try {
            String name = uid.getName(id);
            System.out.println(kind + ' ' + name + ": " + Arrays.toString(id));
            return 0;
        }
        catch (NoSuchUniqueId e) {
            LOG.error(e.getMessage());
            return 1;
        }
    }

    private static byte[] idInBytes(short idwidth, long lid) {
        if (idwidth <= 0) {
            throw new AssertionError((Object)("negative idwidth: " + idwidth));
        }
        byte[] id = Bytes.fromLong((long)lid);
        for (int i = 0; i < id.length - idwidth; ++i) {
            if (id[i] == 0) continue;
            System.err.println(lid + " is too large to fit on " + idwidth + " bytes.  Maybe you forgot to adjust --idwidth?");
            return null;
        }
        return Arrays.copyOfRange(id, id.length - idwidth, id.length);
    }

    private static int lookupName(HBaseClient client, byte[] table, short idwidth, String name, String kind) {
        if (kind != null) {
            return UidManager.extactLookupName(client, table, idwidth, kind, name);
        }
        return UidManager.findAndPrintRow(client, table, CliUtils.toBytes(name), CliUtils.ID_FAMILY, true);
    }

    private static int extactLookupName(HBaseClient client, byte[] table, short idwidth, String kind, String name) {
        UniqueId uid = new UniqueId(client, table, kind, idwidth);
        try {
            byte[] id = uid.getId(name);
            System.out.println(kind + ' ' + name + ": " + Arrays.toString(id));
            return 0;
        }
        catch (NoSuchUniqueName e) {
            LOG.error(e.getMessage());
            return 1;
        }
    }

    private static int metaSync(TSDB tsdb) throws Exception {
        long start_time = System.currentTimeMillis() / 1000L;
        int workers = Runtime.getRuntime().availableProcessors() * 2;
        Set<Integer> processed_tsuids = Collections.synchronizedSet(new HashSet());
        ConcurrentHashMap<String, Long> metric_uids = new ConcurrentHashMap<String, Long>();
        ConcurrentHashMap<String, Long> tagk_uids = new ConcurrentHashMap<String, Long>();
        ConcurrentHashMap<String, Long> tagv_uids = new ConcurrentHashMap<String, Long>();
        List<Scanner> scanners = CliUtils.getDataTableScanners(tsdb, workers);
        LOG.info("Spooling up [" + scanners.size() + "] worker threads");
        ArrayList<MetaSync> threads = new ArrayList<MetaSync>(scanners.size());
        int i = 0;
        for (Scanner scanner : scanners) {
            MetaSync worker = new MetaSync(tsdb, scanner, processed_tsuids, metric_uids, tagk_uids, tagv_uids, i++);
            worker.setName("Sync #" + i);
            worker.start();
            threads.add(worker);
        }
        for (Thread thread : threads) {
            thread.join();
            LOG.info("Thread [" + thread + "] Finished");
        }
        LOG.info("All metasync threads have completed");
        tsdb.flush().joinUninterruptibly();
        long duration = System.currentTimeMillis() / 1000L - start_time;
        LOG.info("Completed meta data synchronization in [" + duration + "] seconds");
        return 0;
    }

    private static int metaPurge(TSDB tsdb) throws Exception {
        int i;
        long start_time = System.currentTimeMillis() / 1000L;
        long max_id = CliUtils.getMaxMetricID(tsdb);
        int workers = Runtime.getRuntime().availableProcessors() * 2;
        double quotient = (double)max_id / (double)workers;
        long index = 1L;
        LOG.info("Max metric ID is [" + max_id + "]");
        LOG.info("Spooling up [" + workers + "] worker threads");
        Thread[] threads = new Thread[workers];
        for (i = 0; i < workers; ++i) {
            threads[i] = new MetaPurge(tsdb, index, quotient, i);
            threads[i].setName("MetaSync # " + i);
            threads[i].start();
            index = (long)((double)index + quotient);
            if (index >= max_id) continue;
            ++index;
        }
        for (i = 0; i < workers; ++i) {
            threads[i].join();
            LOG.info("[" + i + "] Finished");
        }
        tsdb.flush().joinUninterruptibly();
        long duration = System.currentTimeMillis() / 1000L - start_time;
        LOG.info("Completed meta data synchronization in [" + duration + "] seconds");
        return 0;
    }

    private static int treeSync(TSDB tsdb) throws Exception {
        int i;
        long start_time = System.currentTimeMillis() / 1000L;
        long max_id = CliUtils.getMaxMetricID(tsdb);
        int workers = Runtime.getRuntime().availableProcessors() * 2;
        double quotient = (double)max_id / (double)workers;
        long index = 1L;
        LOG.info("Max metric ID is [" + max_id + "]");
        LOG.info("Spooling up [" + workers + "] worker threads");
        Thread[] threads = new Thread[workers];
        for (i = 0; i < workers; ++i) {
            threads[i] = new TreeSync(tsdb, index, quotient, i);
            threads[i].setName("TreeSync # " + i);
            threads[i].start();
            index = (long)((double)index + quotient);
            if (index >= max_id) continue;
            ++index;
        }
        for (i = 0; i < workers; ++i) {
            threads[i].join();
            LOG.info("[" + i + "] Finished");
        }
        tsdb.flush().joinUninterruptibly();
        long duration = System.currentTimeMillis() / 1000L - start_time;
        LOG.info("Completed meta data synchronization in [" + duration + "] seconds");
        return 0;
    }

    private static int purgeTree(TSDB tsdb, int tree_id, boolean delete_definition) throws Exception {
        TreeSync sync = new TreeSync(tsdb, 0L, 1.0, 0);
        return sync.purgeTree(tree_id, delete_definition);
    }
}

