/*
 * Decompiled with CFR 0.152.
 */
package net.opentsdb.tsd;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.stumbleupon.async.Callback;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import net.opentsdb.core.DataPoint;
import net.opentsdb.core.DataPoints;
import net.opentsdb.core.Query;
import net.opentsdb.core.TSDB;
import net.opentsdb.core.TSQuery;
import net.opentsdb.graph.Plot;
import net.opentsdb.meta.Annotation;
import net.opentsdb.stats.Histogram;
import net.opentsdb.stats.StatsCollector;
import net.opentsdb.tsd.BadRequestException;
import net.opentsdb.tsd.GnuplotException;
import net.opentsdb.tsd.HttpQuery;
import net.opentsdb.tsd.HttpRpc;
import net.opentsdb.tsd.QueryRpc;
import net.opentsdb.utils.DateTime;
import net.opentsdb.utils.JSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class GraphHandler
implements HttpRpc {
    private static final Logger LOG = LoggerFactory.getLogger(GraphHandler.class);
    private static final boolean IS_WINDOWS = System.getProperty("os.name", "").contains("Windows");
    private static final AtomicInteger graphs_generated = new AtomicInteger();
    private static final AtomicInteger graphs_diskcache_hit = new AtomicInteger();
    private static final Histogram graphlatency = new Histogram(16000, 2, 100);
    private static final Histogram gnuplotlatency = new Histogram(16000, 2, 100);
    private final ThreadPoolExecutor gnuplot;
    private static final PlotThdFactory thread_factory = new PlotThdFactory();
    private static final String WRAPPER = IS_WINDOWS ? "mygnuplot.bat" : "mygnuplot.sh";
    private static final String GNUPLOT = GraphHandler.findGnuplotHelperScript();

    public GraphHandler() {
        int ncores = Runtime.getRuntime().availableProcessors();
        this.gnuplot = new ThreadPoolExecutor(ncores, ncores, 300000L, TimeUnit.MILLISECONDS, new ArrayBlockingQueue<Runnable>(20 * ncores), thread_factory);
    }

    @Override
    public void execute(TSDB tsdb, HttpQuery query) {
        if (!(query.hasQueryStringParam("json") || query.hasQueryStringParam("png") || query.hasQueryStringParam("ascii"))) {
            String uri = query.request().getUri();
            uri = uri.length() < 4 ? "/" : "/#" + uri.substring(3);
            query.redirect(uri);
            return;
        }
        try {
            this.doGraph(tsdb, query);
        }
        catch (IOException e) {
            query.internalError(e);
        }
        catch (IllegalArgumentException e) {
            query.badRequest(e.getMessage());
        }
    }

    private void doGraph(TSDB tsdb, final HttpQuery query) throws IOException {
        String basepath = this.getGnuplotBasePath(tsdb, query);
        long start_time = DateTime.parseDateTimeString(query.getRequiredQueryStringParam("start"), query.getQueryStringParam("tz"));
        boolean nocache = query.hasQueryStringParam("nocache");
        if (start_time == -1L) {
            throw BadRequestException.missingParameter("start");
        }
        long end_time = DateTime.parseDateTimeString(query.getQueryStringParam("end"), query.getQueryStringParam("tz"));
        long now = System.currentTimeMillis() / 1000L;
        end_time = end_time == -1L ? now : (end_time /= 1000L);
        int max_age = GraphHandler.computeMaxAge(query, start_time /= 1000L, end_time, now);
        if (!nocache && this.isDiskCacheHit(query, end_time, max_age, basepath)) {
            return;
        }
        TSQuery tsquery = QueryRpc.parseQuery(tsdb, query);
        tsquery.validateAndSetQuery();
        Query[] tsdbqueries = tsquery.buildQueries(tsdb);
        List<String> options = query.getQueryStringParams("o");
        if (options == null) {
            options = new ArrayList<String>(tsdbqueries.length);
            for (int i = 0; i < tsdbqueries.length; ++i) {
                options.add("");
            }
        } else {
            if (options.size() != tsdbqueries.length) {
                throw new BadRequestException(options.size() + " `o' parameters, but " + tsdbqueries.length + " `m' parameters.");
            }
            for (String option : options) {
                if (!option.contains("`") && !option.contains("%60") && !option.contains("&#96;")) continue;
                throw new BadRequestException("Option contained a back-tick. That's a no-no.");
            }
        }
        for (Query tsdbquery : tsdbqueries) {
            try {
                tsdbquery.setStartTime(start_time);
            }
            catch (IllegalArgumentException e) {
                throw new BadRequestException("start time: " + e.getMessage());
            }
            try {
                tsdbquery.setEndTime(end_time);
            }
            catch (IllegalArgumentException e) {
                throw new BadRequestException("end time: " + e.getMessage());
            }
        }
        Plot plot = new Plot(start_time, end_time, DateTime.timezones.get(query.getQueryStringParam("tz")));
        GraphHandler.setPlotDimensions(query, plot);
        GraphHandler.setPlotParams(query, plot);
        int nqueries = tsdbqueries.length;
        HashSet[] aggregated_tags = new HashSet[nqueries];
        int npoints = 0;
        for (int i = 0; i < nqueries; ++i) {
            try {
                DataPoints[] series;
                for (DataPoints datapoints : series = tsdbqueries[i].run()) {
                    plot.add(datapoints, options.get(i));
                    aggregated_tags[i] = new HashSet();
                    aggregated_tags[i].addAll(datapoints.getAggregatedTags());
                    npoints += datapoints.aggregatedSize();
                }
            }
            catch (RuntimeException e) {
                GraphHandler.logInfo(query, "Query failed (stack trace coming): " + tsdbqueries[i]);
                throw e;
            }
            tsdbqueries[i] = null;
        }
        tsdbqueries = null;
        if (query.hasQueryStringParam("ascii")) {
            GraphHandler.respondAsciiQuery(query, max_age, basepath, plot);
            return;
        }
        final RunGnuplot rungnuplot = new RunGnuplot(query, max_age, plot, basepath, aggregated_tags, npoints);
        if (!tsquery.getNoAnnotations() && tsquery.getGlobalAnnotations()) {
            class GlobalCB
            implements Callback<Object, List<Annotation>> {
                GlobalCB() {
                }

                public Object call(List<Annotation> global_annotations) throws Exception {
                    rungnuplot.plot.setGlobals(global_annotations);
                    GraphHandler.this.execGnuplot(rungnuplot, query);
                    return null;
                }
            }
            class ErrorCB
            implements Callback<Object, Exception> {
                ErrorCB() {
                }

                public Object call(Exception e) throws Exception {
                    LOG.warn("Failed to retrieve global annotations: ", (Throwable)e);
                    throw e;
                }
            }
            Annotation.getGlobalAnnotations(tsdb, start_time, end_time).addCallback((Callback)new GlobalCB()).addErrback((Callback)new ErrorCB());
        } else {
            this.execGnuplot(rungnuplot, query);
        }
    }

    private void execGnuplot(RunGnuplot rungnuplot, HttpQuery query) {
        try {
            this.gnuplot.execute(rungnuplot);
        }
        catch (RejectedExecutionException e) {
            query.internalError(new Exception("Too many requests pending, please try again later", e));
        }
    }

    private static int computeMaxAge(HttpQuery query, long start_time, long end_time, long now) {
        if (end_time > now) {
            return 0;
        }
        if (end_time < now - 3600L && !DateTime.isRelativeDate(query.getQueryStringParam("start")) && !DateTime.isRelativeDate(query.getQueryStringParam("end"))) {
            return 86400;
        }
        return (int)(end_time - start_time) >> 10;
    }

    public void shutdown() {
        this.gnuplot.shutdown();
    }

    public static void collectStats(StatsCollector collector) {
        collector.record("http.latency", graphlatency, "type=graph");
        collector.record("http.latency", gnuplotlatency, "type=gnuplot");
        collector.record("http.graph.requests", graphs_diskcache_hit, "cache=disk");
        collector.record("http.graph.requests", graphs_generated, "cache=miss");
    }

    private String getGnuplotBasePath(TSDB tsdb, HttpQuery query) {
        Map<String, List<String>> q = query.getQueryString();
        q.remove("ignore");
        HashMap<String, List<String>> qs = new HashMap<String, List<String>>(q);
        qs.remove("png");
        qs.remove("json");
        qs.remove("ascii");
        return tsdb.getConfig().getDirectoryName("tsd.http.cachedir") + Integer.toHexString(qs.hashCode());
    }

    private boolean isDiskCacheHit(HttpQuery query, long end_time, int max_age, String basepath) throws IOException {
        String cachepath = basepath + (query.hasQueryStringParam("ascii") ? ".txt" : ".png");
        File cachedfile = new File(cachepath);
        if (cachedfile.exists()) {
            long bytes = cachedfile.length();
            if (bytes < 21L) {
                GraphHandler.logWarn(query, "Cached " + cachepath + " is too small (" + bytes + " bytes) to be valid.  Ignoring it.");
                return false;
            }
            if (GraphHandler.staleCacheFile(query, end_time, max_age, cachedfile)) {
                return false;
            }
            if (query.hasQueryStringParam("json")) {
                HashMap<String, Object> map = this.loadCachedJson(query, end_time, max_age, basepath);
                if (map == null) {
                    map = new HashMap();
                }
                map.put("timing", query.processingTimeMillis());
                map.put("cachehit", "disk");
                query.sendReply(JSON.serializeToBytes(map));
            } else if (query.hasQueryStringParam("png") || query.hasQueryStringParam("ascii")) {
                query.sendFile(cachepath, max_age);
            } else {
                query.sendReply(HttpQuery.makePage("TSDB Query", "Your graph is ready", "<img src=\"" + query.request().getUri() + "&amp;png\"/><br/><small>(served from disk cache)</small>"));
            }
            graphs_diskcache_hit.incrementAndGet();
            return true;
        }
        HashMap<String, Object> map = this.loadCachedJson(query, end_time, max_age, basepath);
        if (map == null || !map.containsKey("plotted") || (Integer)map.get("plotted") == 0) {
            return false;
        }
        if (query.hasQueryStringParam("json")) {
            map.put("timing", query.processingTimeMillis());
            map.put("cachehit", "disk");
            query.sendReply(JSON.serializeToBytes(map));
        } else if (query.hasQueryStringParam("png")) {
            query.sendReply(" ");
        } else {
            query.sendReply(HttpQuery.makePage("TSDB Query", "No results", "Sorry, your query didn't return anything.<br/><small>(served from disk cache)</small>"));
        }
        graphs_diskcache_hit.incrementAndGet();
        return true;
    }

    private static boolean staleCacheFile(HttpQuery query, long end_time, long max_age, File cachedfile) {
        long mtime = cachedfile.lastModified() / 1000L;
        if (mtime <= 0L) {
            return true;
        }
        long now = System.currentTimeMillis() / 1000L;
        long staleness = now - mtime;
        if (staleness < 0L) {
            GraphHandler.logWarn(query, "Not using file @ " + cachedfile + " with weird mtime in the future: " + mtime);
            return true;
        }
        if (0L < end_time && end_time < now) {
            return mtime < end_time;
        }
        if (staleness > max_age) {
            GraphHandler.logInfo(query, "Cached file @ " + cachedfile.getPath() + " is " + staleness + "s stale, which is more than its limit of " + max_age + "s, and needs to be regenerated.");
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeFile(HttpQuery query, String path, byte[] contents) {
        try {
            FileOutputStream out = new FileOutputStream(path);
            try {
                out.write(contents);
            }
            finally {
                out.close();
            }
        }
        catch (FileNotFoundException e) {
            GraphHandler.logError(query, "Failed to create file " + path, e);
        }
        catch (IOException e) {
            GraphHandler.logError(query, "Failed to write file " + path, e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static byte[] readFile(HttpQuery query, File file, int max_length) {
        FileInputStream in;
        int length = (int)file.length();
        if (length <= 0) {
            return null;
        }
        try {
            in = new FileInputStream(file.getPath());
        }
        catch (FileNotFoundException e) {
            return null;
        }
        try {
            byte[] buf = new byte[Math.min(length, max_length)];
            int read = in.read(buf);
            if (read != buf.length) {
                GraphHandler.logError(query, "When reading " + file + ": read only " + read + " bytes instead of " + buf.length);
                byte[] byArray = null;
                return byArray;
            }
            byte[] byArray = buf;
            return byArray;
        }
        catch (IOException e) {
            GraphHandler.logError(query, "Error while reading " + file, e);
            byte[] byArray = null;
            return byArray;
        }
        finally {
            try {
                in.close();
            }
            catch (IOException e) {
                GraphHandler.logError(query, "Error while closing " + file, e);
            }
        }
    }

    private HashMap<String, Object> loadCachedJson(HttpQuery query, long end_time, long max_age, String basepath) throws JsonParseException, JsonMappingException, IOException {
        String json_path = basepath + ".json";
        File json_cache = new File(json_path);
        if (GraphHandler.staleCacheFile(query, end_time, max_age, json_cache)) {
            return null;
        }
        byte[] json = GraphHandler.readFile(query, json_cache, 4096);
        if (json == null) {
            return null;
        }
        json_cache = null;
        return JSON.parseToObject(json, HashMap.class);
    }

    static void setPlotDimensions(HttpQuery query, Plot plot) {
        String wxh = query.getQueryStringParam("wxh");
        if (wxh != null && !wxh.isEmpty()) {
            if (wxh.contains("`") || wxh.contains("%60") || wxh.contains("&#96;")) {
                throw new BadRequestException("WXH contained a back-tick. That's a no-no.");
            }
            int wxhlength = wxh.length();
            if (wxhlength < 7) {
                throw new BadRequestException("Parameter wxh too short: " + wxh);
            }
            int x = wxh.indexOf(120, 3);
            if (x < 0) {
                throw new BadRequestException("Invalid wxh parameter: " + wxh);
            }
            try {
                short width = Short.parseShort(wxh.substring(0, x));
                short height = Short.parseShort(wxh.substring(x + 1, wxhlength));
                try {
                    plot.setDimensions(width, height);
                }
                catch (IllegalArgumentException e) {
                    throw new BadRequestException("Invalid wxh parameter: " + wxh + ", " + e.getMessage());
                }
            }
            catch (NumberFormatException e) {
                throw new BadRequestException("Can't parse wxh '" + wxh + "': " + e.getMessage());
            }
        }
    }

    private static String stringify(String s) {
        StringBuilder buf = new StringBuilder(1 + s.length() + 1);
        buf.append('\"');
        HttpQuery.escapeJson(s, buf);
        buf.append('\"');
        return buf.toString();
    }

    private static String popParam(Map<String, List<String>> querystring, String param) {
        List<String> params = querystring.remove(param);
        if (params == null) {
            return null;
        }
        String given = params.get(params.size() - 1);
        if (given.contains("`") || given.contains("%60") || given.contains("&#96;")) {
            throw new BadRequestException("Parameter " + param + " contained a back-tick. That's a no-no.");
        }
        return given;
    }

    static void setPlotParams(HttpQuery query, Plot plot) {
        HashMap<String, String> params = new HashMap<String, String>();
        Map<String, List<String>> querystring = query.getQueryString();
        String value = GraphHandler.popParam(querystring, "yrange");
        if (value != null) {
            params.put("yrange", value);
        }
        if ((value = GraphHandler.popParam(querystring, "y2range")) != null) {
            params.put("y2range", value);
        }
        if ((value = GraphHandler.popParam(querystring, "ylabel")) != null) {
            params.put("ylabel", GraphHandler.stringify(value));
        }
        if ((value = GraphHandler.popParam(querystring, "y2label")) != null) {
            params.put("y2label", GraphHandler.stringify(value));
        }
        if ((value = GraphHandler.popParam(querystring, "yformat")) != null) {
            params.put("format y", GraphHandler.stringify(value));
        }
        if ((value = GraphHandler.popParam(querystring, "y2format")) != null) {
            params.put("format y2", GraphHandler.stringify(value));
        }
        if ((value = GraphHandler.popParam(querystring, "xformat")) != null) {
            params.put("format x", GraphHandler.stringify(value));
        }
        if ((value = GraphHandler.popParam(querystring, "ylog")) != null) {
            params.put("logscale y", "");
        }
        if ((value = GraphHandler.popParam(querystring, "y2log")) != null) {
            params.put("logscale y2", "");
        }
        if ((value = GraphHandler.popParam(querystring, "key")) != null) {
            params.put("key", value);
        }
        if ((value = GraphHandler.popParam(querystring, "title")) != null) {
            params.put("title", GraphHandler.stringify(value));
        }
        if ((value = GraphHandler.popParam(querystring, "bgcolor")) != null) {
            params.put("bgcolor", value);
        }
        if ((value = GraphHandler.popParam(querystring, "fgcolor")) != null) {
            params.put("fgcolor", value);
        }
        if ((value = GraphHandler.popParam(querystring, "smooth")) != null) {
            params.put("smooth", value);
        }
        if ((value = GraphHandler.popParam(querystring, "style")) != null) {
            params.put("style", value);
        }
        if ((value = GraphHandler.popParam(querystring, "nokey")) != null) {
            params.put("key", null);
        }
        plot.setParams(params);
    }

    static int runGnuplot(HttpQuery query, String basepath, Plot plot) throws IOException {
        int rv;
        int nplotted = plot.dumpToFiles(basepath);
        long start_time = System.nanoTime();
        Process gnuplot = new ProcessBuilder(GNUPLOT, basepath + ".out", basepath + ".err", basepath + ".gnuplot").start();
        try {
            rv = gnuplot.waitFor();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("interrupted", e);
        }
        finally {
            gnuplot.destroy();
        }
        gnuplotlatency.add((int)((System.nanoTime() - start_time) / 1000000L));
        if (rv != 0) {
            byte[] stderr = GraphHandler.readFile(query, new File(basepath + ".err"), 4096);
            new File(basepath + ".png").delete();
            if (stderr == null) {
                throw new GnuplotException(rv);
            }
            throw new GnuplotException(new String(stderr));
        }
        GraphHandler.deleteFileIfEmpty(basepath + ".out");
        GraphHandler.deleteFileIfEmpty(basepath + ".err");
        return nplotted;
    }

    private static void deleteFileIfEmpty(String path) {
        File file = new File(path);
        if (file.length() <= 0L) {
            file.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void respondAsciiQuery(HttpQuery query, int max_age, String basepath, Plot plot) {
        PrintWriter asciifile;
        String path = basepath + ".txt";
        try {
            asciifile = new PrintWriter(path);
        }
        catch (IOException e) {
            query.internalError(e);
            return;
        }
        try {
            StringBuilder tagbuf = new StringBuilder();
            for (DataPoints dp : plot.getDataPoints()) {
                String metric = dp.metricName();
                tagbuf.setLength(0);
                for (Map.Entry<String, String> tag : dp.getTags().entrySet()) {
                    tagbuf.append(' ').append(tag.getKey()).append('=').append(tag.getValue());
                }
                for (DataPoint d : dp) {
                    if (d.isInteger()) {
                        GraphHandler.printMetricHeader(asciifile, metric, d.timestamp());
                        asciifile.print(d.longValue());
                    } else {
                        double value = d.doubleValue();
                        if (Double.isInfinite(value)) {
                            throw new IllegalStateException("Infinity:" + value + " d=" + d + ", query=" + query);
                        }
                        if (Double.isNaN(value)) continue;
                        GraphHandler.printMetricHeader(asciifile, metric, d.timestamp());
                        asciifile.print(value);
                    }
                    asciifile.print(tagbuf);
                    asciifile.print('\n');
                }
            }
        }
        finally {
            asciifile.close();
        }
        try {
            query.sendFile(path, max_age);
        }
        catch (IOException e) {
            query.internalError(e);
        }
    }

    private static void printMetricHeader(PrintWriter writer, String metric, long timestamp) {
        writer.print(metric);
        writer.print(' ');
        writer.print(timestamp / 1000L);
        writer.print(' ');
    }

    private static String findGnuplotHelperScript() {
        String error;
        URL url = GraphHandler.class.getClassLoader().getResource(WRAPPER);
        if (url == null) {
            throw new RuntimeException("Couldn't find " + WRAPPER + " on the CLASSPATH: " + System.getProperty("java.class.path"));
        }
        String path = url.getFile();
        LOG.debug("Using Gnuplot wrapper at {}", (Object)path);
        File file = new File(path);
        if (!file.exists()) {
            error = "non-existent";
        } else if (!file.canExecute()) {
            error = "non-executable";
        } else if (!file.canRead()) {
            error = "unreadable";
        } else {
            return path;
        }
        throw new RuntimeException("The " + WRAPPER + " found on the CLASSPATH (" + path + ") is a " + error + " file...  WTF?  CLASSPATH=" + System.getProperty("java.class.path"));
    }

    static void logInfo(HttpQuery query, String msg) {
        LOG.info(query.channel().toString() + ' ' + msg);
    }

    static void logWarn(HttpQuery query, String msg) {
        LOG.warn(query.channel().toString() + ' ' + msg);
    }

    static void logError(HttpQuery query, String msg) {
        LOG.error(query.channel().toString() + ' ' + msg);
    }

    static void logError(HttpQuery query, String msg, Throwable e) {
        LOG.error(query.channel().toString() + ' ' + msg, e);
    }

    private static final class PlotThdFactory
    implements ThreadFactory {
        private final AtomicInteger id = new AtomicInteger(0);

        private PlotThdFactory() {
        }

        @Override
        public Thread newThread(Runnable r) {
            return new Thread(r, "Gnuplot #" + this.id.incrementAndGet());
        }
    }

    private static final class RunGnuplot
    implements Runnable {
        private final HttpQuery query;
        private final int max_age;
        private final Plot plot;
        private final String basepath;
        private final HashSet<String>[] aggregated_tags;
        private final int npoints;

        public RunGnuplot(HttpQuery query, int max_age, Plot plot, String basepath, HashSet<String>[] aggregated_tags, int npoints) {
            this.query = query;
            this.max_age = max_age;
            this.plot = plot;
            this.basepath = IS_WINDOWS ? basepath.replace("\\", "\\\\").replace("/", "\\\\") : basepath;
            this.aggregated_tags = aggregated_tags;
            this.npoints = npoints;
        }

        @Override
        public void run() {
            try {
                this.execute();
            }
            catch (BadRequestException e) {
                this.query.badRequest(e.getMessage());
            }
            catch (GnuplotException e) {
                this.query.badRequest("<pre>" + e.getMessage() + "</pre>");
            }
            catch (RuntimeException e) {
                this.query.internalError(e);
            }
            catch (IOException e) {
                this.query.internalError(e);
            }
        }

        private void execute() throws IOException {
            int nplotted = GraphHandler.runGnuplot(this.query, this.basepath, this.plot);
            if (this.query.hasQueryStringParam("json")) {
                HashMap<String, Object> results = new HashMap<String, Object>();
                results.put("plotted", nplotted);
                results.put("points", this.npoints);
                if (this.aggregated_tags != null && this.aggregated_tags.length > 0 && this.aggregated_tags[0] == null) {
                    this.aggregated_tags[0] = new HashSet();
                }
                results.put("etags", this.aggregated_tags);
                results.put("timing", this.query.processingTimeMillis());
                this.query.sendReply(JSON.serializeToBytes(results));
                GraphHandler.writeFile(this.query, this.basepath + ".json", JSON.serializeToBytes(results));
            } else if (this.query.hasQueryStringParam("png")) {
                this.query.sendFile(this.basepath + ".png", this.max_age);
            } else {
                this.query.internalError(new Exception("Should never be here!"));
            }
            graphlatency.add(this.query.processingTimeMillis());
            graphs_generated.incrementAndGet();
        }
    }
}

