/*
 * Decompiled with CFR 0.152.
 */
package dev.miku.r2dbc.mysql;

import dev.miku.r2dbc.mysql.ConnectionContext;
import dev.miku.r2dbc.mysql.MySqlBatch;
import dev.miku.r2dbc.mysql.MySqlBatchingBatch;
import dev.miku.r2dbc.mysql.MySqlConnectionMetadata;
import dev.miku.r2dbc.mysql.MySqlResult;
import dev.miku.r2dbc.mysql.MySqlStatement;
import dev.miku.r2dbc.mysql.MySqlSyntheticBatch;
import dev.miku.r2dbc.mysql.PrepareParametrizedStatement;
import dev.miku.r2dbc.mysql.PrepareQuery;
import dev.miku.r2dbc.mysql.PrepareSimpleStatement;
import dev.miku.r2dbc.mysql.Query;
import dev.miku.r2dbc.mysql.QueryFlow;
import dev.miku.r2dbc.mysql.ServerVersion;
import dev.miku.r2dbc.mysql.SimpleQuery;
import dev.miku.r2dbc.mysql.TextParametrizedStatement;
import dev.miku.r2dbc.mysql.TextQuery;
import dev.miku.r2dbc.mysql.TextSimpleStatement;
import dev.miku.r2dbc.mysql.client.Client;
import dev.miku.r2dbc.mysql.codec.Codecs;
import dev.miku.r2dbc.mysql.message.client.PingMessage;
import dev.miku.r2dbc.mysql.message.server.CompleteMessage;
import dev.miku.r2dbc.mysql.message.server.ErrorMessage;
import dev.miku.r2dbc.mysql.message.server.ServerMessage;
import dev.miku.r2dbc.mysql.util.AssertUtils;
import io.netty.util.ReferenceCountUtil;
import io.r2dbc.spi.Connection;
import io.r2dbc.spi.IsolationLevel;
import io.r2dbc.spi.R2dbcException;
import io.r2dbc.spi.ValidationDepth;
import java.time.DateTimeException;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Predicate;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;
import reactor.core.publisher.SynchronousSink;
import reactor.util.annotation.Nullable;

public final class MySqlConnection
implements Connection {
    private static final Logger logger = LoggerFactory.getLogger(MySqlConnection.class);
    private static final String ZONE_PREFIX_POSIX = "posix/";
    private static final String ZONE_PREFIX_RIGHT = "right/";
    private static final int PREFIX_LENGTH = 6;
    private static final ServerVersion TRAN_LEVEL_8X = ServerVersion.create(8, 0, 3);
    private static final ServerVersion TRAN_LEVEL_5X = ServerVersion.create(5, 7, 20);
    private static final ServerVersion TX_LEVEL_8X = ServerVersion.create(8, 0, 0);
    private static final Predicate<ServerMessage> PING_DONE = message -> message instanceof ErrorMessage || message instanceof CompleteMessage && ((CompleteMessage)message).isDone();
    private static final Function<MySqlResult, Publisher<InitData>> INIT_HANDLER = r -> r.map((row, meta) -> new InitData(MySqlConnection.convertIsolationLevel((String)row.get(0, String.class)), (String)row.get(1, String.class), null));
    private static final Function<MySqlResult, Publisher<InitData>> FULL_INIT_HANDLER = r -> r.map((row, meta) -> {
        ZoneId zoneId;
        IsolationLevel level = MySqlConnection.convertIsolationLevel((String)row.get(0, String.class));
        String product = (String)row.get(1, String.class);
        String systemTimeZone = (String)row.get(2, String.class);
        String timeZone = (String)row.get(3, String.class);
        if (timeZone == null || timeZone.isEmpty() || "SYSTEM".equals(timeZone)) {
            if (systemTimeZone == null || systemTimeZone.isEmpty()) {
                logger.warn("MySQL does not return any timezone, trying to use system default timezone");
                zoneId = ZoneId.systemDefault();
            } else {
                zoneId = MySqlConnection.convertZoneId(systemTimeZone);
            }
        } else {
            zoneId = MySqlConnection.convertZoneId(timeZone);
        }
        return new InitData(level, product, zoneId);
    });
    private static final BiConsumer<ServerMessage, SynchronousSink<Boolean>> PING_HANDLER = (message, sink) -> {
        if (message instanceof ErrorMessage) {
            ErrorMessage msg = (ErrorMessage)message;
            logger.debug("Remote validate failed: [{}] [{}] {}", new Object[]{msg.getErrorCode(), msg.getSqlState(), msg.getErrorMessage()});
            sink.next((Object)false);
        } else if (message instanceof CompleteMessage && ((CompleteMessage)message).isDone()) {
            sink.next((Object)true);
        } else {
            ReferenceCountUtil.safeRelease((Object)message);
        }
    };
    private final Client client;
    private final Codecs codecs;
    private final boolean batchSupported;
    private final ConnectionContext context;
    private final MySqlConnectionMetadata metadata;
    private final IsolationLevel sessionLevel;
    @Nullable
    private final Predicate<String> prepare;
    private volatile IsolationLevel currentLevel;

    MySqlConnection(Client client, ConnectionContext context, Codecs codecs, IsolationLevel level, @Nullable String product, @Nullable Predicate<String> prepare) {
        this.client = client;
        this.context = context;
        this.sessionLevel = level;
        this.currentLevel = level;
        this.codecs = codecs;
        this.metadata = new MySqlConnectionMetadata(context.getServerVersion().toString(), product);
        this.batchSupported = (context.getCapabilities() & 0x10000) != 0;
        this.prepare = prepare;
        if (this.batchSupported) {
            logger.debug("Batch is supported by server");
        } else {
            logger.warn("The MySQL server does not support batch executing, fallback to executing one-by-one");
        }
    }

    public Mono<Void> beginTransaction() {
        return Mono.defer(() -> {
            if (this.isInTransaction()) {
                return Mono.empty();
            }
            return QueryFlow.executeVoid(this.client, "BEGIN");
        });
    }

    public Mono<Void> close() {
        Mono<Void> closer = this.client.close();
        if (logger.isDebugEnabled()) {
            return closer.doOnSubscribe(s -> logger.debug("Connection closing")).doOnSuccess(ignored -> logger.debug("Connection close succeed"));
        }
        return closer;
    }

    public Mono<Void> commitTransaction() {
        return Mono.defer(() -> {
            if (!this.isInTransaction()) {
                return Mono.empty();
            }
            return this.recoverIsolationLevel(QueryFlow.executeVoid(this.client, "COMMIT"));
        });
    }

    public MySqlBatch createBatch() {
        if (this.batchSupported) {
            return new MySqlBatchingBatch(this.client, this.codecs, this.context);
        }
        return new MySqlSyntheticBatch(this.client, this.codecs, this.context);
    }

    public Mono<Void> createSavepoint(String name) {
        AssertUtils.requireValidName(name, "Savepoint name must not be empty and not contain backticks");
        String sql = String.format("SAVEPOINT `%s`", name);
        return Mono.defer(() -> {
            if (this.isInTransaction()) {
                return QueryFlow.executeVoid(this.client, sql);
            }
            if (this.batchSupported) {
                return QueryFlow.executeVoid(this.client, "BEGIN;" + sql);
            }
            return QueryFlow.executeVoid(this.client, "BEGIN", sql);
        });
    }

    public MySqlStatement createStatement(String sql) {
        AssertUtils.requireNonNull(sql, "sql must not be null");
        Query query = Query.parse(sql, this.prepare != null);
        if (query instanceof SimpleQuery) {
            if (this.prepare != null && this.prepare.test(sql)) {
                logger.debug("Create a simple statement provided by prepare query");
                return new PrepareSimpleStatement(this.client, this.codecs, this.context, sql);
            }
            logger.debug("Create a simple statement provided by text query");
            return new TextSimpleStatement(this.client, this.codecs, this.context, sql);
        }
        if (query instanceof TextQuery) {
            logger.debug("Create a parametrized statement provided by text query");
            return new TextParametrizedStatement(this.client, this.codecs, this.context, (TextQuery)query);
        }
        logger.debug("Create a parametrized statement provided by prepare query");
        return new PrepareParametrizedStatement(this.client, this.codecs, this.context, (PrepareQuery)query);
    }

    public Mono<Void> releaseSavepoint(String name) {
        AssertUtils.requireValidName(name, "Savepoint name must not be empty and not contain backticks");
        return QueryFlow.executeVoid(this.client, String.format("RELEASE SAVEPOINT `%s`", name));
    }

    public Mono<Void> rollbackTransaction() {
        return Mono.defer(() -> {
            if (!this.isInTransaction()) {
                return Mono.empty();
            }
            return this.recoverIsolationLevel(QueryFlow.executeVoid(this.client, "ROLLBACK"));
        });
    }

    public Mono<Void> rollbackTransactionToSavepoint(String name) {
        AssertUtils.requireValidName(name, "Savepoint name must not be empty and not contain backticks");
        return QueryFlow.executeVoid(this.client, String.format("ROLLBACK TO SAVEPOINT `%s`", name));
    }

    public MySqlConnectionMetadata getMetadata() {
        return this.metadata;
    }

    public IsolationLevel getTransactionIsolationLevel() {
        return this.currentLevel;
    }

    public Mono<Void> setTransactionIsolationLevel(IsolationLevel isolationLevel) {
        AssertUtils.requireNonNull(isolationLevel, "isolationLevel must not be null");
        return QueryFlow.executeVoid(this.client, String.format("SET TRANSACTION ISOLATION LEVEL %s", isolationLevel.asSql())).doOnSuccess(ignored -> {
            this.currentLevel = isolationLevel;
        });
    }

    public Mono<Boolean> validate(ValidationDepth depth) {
        AssertUtils.requireNonNull(depth, "depth must not be null");
        if (depth == ValidationDepth.LOCAL) {
            return Mono.fromSupplier(this.client::isConnected);
        }
        return Mono.defer(() -> {
            if (!this.client.isConnected()) {
                return Mono.just((Object)false);
            }
            return this.client.exchange(PingMessage.getInstance(), PING_DONE).handle(PING_HANDLER).last().onErrorResume(e -> {
                logger.debug("Remote validate failed", e);
                return Mono.just((Object)false);
            });
        });
    }

    public boolean isAutoCommit() {
        return !this.isInTransaction() && this.isSessionAutoCommit();
    }

    public Mono<Void> setAutoCommit(boolean autoCommit) {
        return Mono.defer(() -> {
            if (autoCommit == this.isSessionAutoCommit()) {
                return Mono.empty();
            }
            return QueryFlow.executeVoid(this.client, String.format("SET autocommit=%d", autoCommit ? 1 : 0));
        });
    }

    boolean isInTransaction() {
        return (this.context.getServerStatuses() & 1) != 0;
    }

    boolean isSessionAutoCommit() {
        return (this.context.getServerStatuses() & 2) != 0;
    }

    private Mono<Void> recoverIsolationLevel(Mono<Void> commitOrRollback) {
        if (this.currentLevel != this.sessionLevel) {
            return commitOrRollback.doOnSuccess(ignored -> {
                this.currentLevel = this.sessionLevel;
            }).doOnError(e -> {
                if (e instanceof R2dbcException) {
                    this.currentLevel = this.sessionLevel;
                }
            });
        }
        return commitOrRollback;
    }

    static Mono<MySqlConnection> init(Client client, Codecs codecs, ConnectionContext context, @Nullable Predicate<String> prepare) {
        Function<MySqlResult, Publisher<InitData>> handler;
        ServerVersion version = context.getServerVersion();
        StringBuilder query = new StringBuilder(128);
        if (version.isGreaterThanOrEqualTo(TRAN_LEVEL_8X) || version.isGreaterThanOrEqualTo(TRAN_LEVEL_5X) && version.isLessThan(TX_LEVEL_8X)) {
            query.append("SELECT @@transaction_isolation AS i, @@version_comment AS v");
        } else {
            query.append("SELECT @@tx_isolation AS i, @@version_comment AS v");
        }
        if (context.shouldSetServerZoneId()) {
            handler = FULL_INIT_HANDLER;
            query.append(", @@system_time_zone AS s, @@time_zone AS t");
        } else {
            handler = INIT_HANDLER;
        }
        return new TextSimpleStatement(client, codecs, context, query.toString()).execute().flatMap(handler).last().map(data -> {
            ZoneId serverZoneId = ((InitData)data).serverZoneId;
            if (serverZoneId != null) {
                logger.debug("Set server time zone to {} from init query", (Object)serverZoneId);
                context.setServerZoneId(serverZoneId);
            }
            return new MySqlConnection(client, context, codecs, ((InitData)data).level, ((InitData)data).product, prepare);
        });
    }

    private static ZoneId convertZoneId(String id) {
        String realId = id.startsWith(ZONE_PREFIX_POSIX) || id.startsWith(ZONE_PREFIX_RIGHT) ? id.substring(6) : id;
        try {
            switch (realId) {
                case "Factory": {
                    return ZoneOffset.UTC;
                }
                case "America/Nuuk": {
                    return ZoneId.of("America/Godthab");
                }
                case "ROC": {
                    return ZoneId.of("+8");
                }
            }
            return ZoneId.of(realId, ZoneId.SHORT_IDS);
        }
        catch (DateTimeException e) {
            logger.warn("The server timezone is <{}> that's unknown, trying to use system default timezone", (Object)id);
            return ZoneId.systemDefault();
        }
    }

    private static IsolationLevel convertIsolationLevel(@Nullable String name) {
        if (name == null) {
            logger.warn("Isolation level is null in current session, fallback to repeatable read");
            return IsolationLevel.REPEATABLE_READ;
        }
        switch (name) {
            case "READ-UNCOMMITTED": {
                return IsolationLevel.READ_UNCOMMITTED;
            }
            case "READ-COMMITTED": {
                return IsolationLevel.READ_COMMITTED;
            }
            case "REPEATABLE-READ": {
                return IsolationLevel.REPEATABLE_READ;
            }
            case "SERIALIZABLE": {
                return IsolationLevel.SERIALIZABLE;
            }
        }
        logger.warn("Unknown isolation level {} in current session, fallback to repeatable read", (Object)name);
        return IsolationLevel.REPEATABLE_READ;
    }

    private static class InitData {
        private final IsolationLevel level;
        @Nullable
        private final String product;
        @Nullable
        private final ZoneId serverZoneId;

        private InitData(IsolationLevel level, @Nullable String product, @Nullable ZoneId serverZoneId) {
            this.level = level;
            this.product = product;
            this.serverZoneId = serverZoneId;
        }
    }
}

