package com.vcarecity.gbtparser.service.impl;

import com.vcarecity.gbt.constant.CmdConstant;
import com.vcarecity.gbt.constant.ConstantKey;
import com.vcarecity.gbt.context.AppDataMessageData;
import com.vcarecity.gbt.context.BaseMessageData;
import com.vcarecity.gbt.exception.MissingTypFlagJsonKeyException;
import com.vcarecity.gbt.exception.TypeFlagNotExistException;
import com.vcarecity.gbt.status.ConvertStatus;
import com.vcarecity.gbt.util.CheckUtil;
import com.vcarecity.gbt.util.HexUtil;
import com.vcarecity.gbtparser.result.ConvertResult;
import com.vcarecity.gbtparser.result.ConvertResultHelper;
import com.vcarecity.gbtparser.service.IGbtConvertService;
import com.vcarecity.gbtparser.typeflag.ITypeFlagConvert;
import com.vcarecity.gbtparser.util.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;

import static com.vcarecity.gbtparser.GbtApplication.TYPE_FLAG_CONVERTER;

/**
 * @author Kerry on 18/08/16
 */

public class GbtConvertServiceImpl implements IGbtConvertService<BaseMessageData<AppDataMessageData>> {
    private static Logger logger = LoggerFactory.getLogger(GbtConvertServiceImpl.class);

    @NotNull
    @Override
    public ConvertResult convertObject(BaseMessageData<AppDataMessageData> message) {

        byte[] serialNum = HexUtil.intToByteArray(message.getSerialNum(), 2);
        byte[] version = HexUtil.intToByteArray(message.getVersion(), 2);
        byte[] date;
        if (StringUtils.isEmpty(message.getDate())) {
            date = HexUtil.hexToByteArray(LocalDateTime.now().format(DateTimeFormatter.ofPattern(ConstantKey.GBT_DATE_FORMAT)));
        } else {
            date = HexUtil.hexToByteArray(message.getDate());
        }
        // require
        assert date.length == 6;

        byte[] sourceAddress = HexUtil.hexToByteArray(message.getSourceAddress());
        assert sourceAddress.length == 6;

        byte[] destAddress = HexUtil.hexToByteArray(message.getDestAddress());
        assert destAddress.length == 6;


        switch (message.getCmd()) {
            case CmdConstant.CMD_CONTROL:
                //下行 控制指令
                break;
            case CmdConstant.CMD_REQUEST:
                //下行 请求指令
                break;
            default:
                //下行 其他指令都是不合法
                return ConvertResultHelper.build(ConvertStatus.CMD_ILLEGAL, message);
        }
        byte[] appData;
        try {
            appData = convertAppData(message.getAppData());
        } catch (TypeFlagNotExistException e) {
            logger.warn(e.getMessage(), e);
            e.printStackTrace();
            return ConvertResultHelper.build(ConvertStatus.CONVERT_NOT_FOUNT, message);
        } catch (MissingTypFlagJsonKeyException e) {
            logger.warn(e.getMessage(), e);
            e.printStackTrace();
            return ConvertResultHelper.build(ConvertStatus.MISSING_JSON_KEY, message);
        }

        byte[] appDataLen = HexUtil.intToByteArray(appData.length, 2);
        byte[] cmd = HexUtil.intToByteArray(message.getCmd(), 1);

        byte[] result = HexUtil.concatAll(serialNum, version, date, sourceAddress, destAddress, appDataLen, cmd, appData);
        byte[] calcCrc = CheckUtil.calcCrc(result);

        return ConvertResultHelper.ok(message, HexUtil.concatAll(result, calcCrc));
    }

    /**
     * @param appData
     * @return
     * @throws TypeFlagNotExistException
     * @throws MissingTypFlagJsonKeyException
     */
    private byte[] convertAppData(AppDataMessageData appData) throws TypeFlagNotExistException, MissingTypFlagJsonKeyException {

        final int typeFlag = appData.getTypeFlag();

        ITypeFlagConvert typeFlagConverter = newTypeFlagInstance(typeFlag);
        if (typeFlagConverter == null) {
            throw new TypeFlagNotExistException(typeFlag);
        }

        List<Map<String, Object>> dataInfo = appData.getDataInfo();

        byte[] dimension = new byte[0];

        if (dataInfo != null && !dataInfo.isEmpty()) {
            Integer infoObjectCount = typeFlagConverter.getInfoObjectCount();
            if (infoObjectCount == null) {
                infoObjectCount = dataInfo.size();
            }
            byte[][] appBody = new byte[infoObjectCount][];
            int index = 0;
            for (Map<String, Object> map : dataInfo) {
                byte[] temp = typeFlagConverter.convertInfoObjectItem(typeFlag, map);
                if (temp != null && temp.length > 0) {
                    appBody[index++] = temp;
                }
            }
            dimension = HexUtil.concatAll(HexUtil.intToByteArray(index, 1), HexUtil.reduceDimension(appBody));
        }
        return HexUtil.concatAll(HexUtil.intToByteArray(typeFlag, 1), dimension);
    }

    private ITypeFlagConvert newTypeFlagInstance(int typeFlag) {
        Class<? extends ITypeFlagConvert> currentConvertClazz = TYPE_FLAG_CONVERTER.get(typeFlag);
        if (currentConvertClazz == null) {
            return null;
        }
        try {
            return currentConvertClazz.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            logger.error(e.getMessage(), e);
            e.printStackTrace();
        }
        return null;
    }


}