package com.vcarecity.gbtparser.service.impl;

import com.vcarecity.gbt.config.GbtGlobalConfig;
import com.vcarecity.gbt.constant.CmdConstant;
import com.vcarecity.gbt.context.AppDataMessageData;
import com.vcarecity.gbt.context.BaseMessageData;
import com.vcarecity.gbt.exception.TypeFlagNotExistException;
import com.vcarecity.gbt.status.ParserStatus;
import com.vcarecity.gbt.util.CheckUtil;
import com.vcarecity.gbt.util.HexUtil;
import com.vcarecity.gbtparser.result.ParserResult;
import com.vcarecity.gbtparser.result.ParserResultHelper;
import com.vcarecity.gbtparser.service.IGbtParserService;
import com.vcarecity.gbtparser.typeflag.ITypeFlagParser;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static com.vcarecity.gbtparser.GbtApplication.TYPE_FLAG_PARSER;

/**
 * @author Kerry on 18/08/16
 */

public class GbtParserServiceImpl implements IGbtParserService<BaseMessageData<AppDataMessageData>> {

    private static Logger logger = LoggerFactory.getLogger(GbtParserServiceImpl.class);

    /**
     * 解析
     *
     * @param data
     * @return 解析结果
     * @throws RuntimeException 否认
     */
    @NotNull
    @Override
    public ParserResult<BaseMessageData<AppDataMessageData>> parserFrame(byte[] data) {

        int allLen = data.length;
        byte[] crcByte = HexUtil.copyTo(data, allLen - 1, 1);

        data = Arrays.copyOf(data, allLen - 1);

        byte[] calcCrc = CheckUtil.calcCrc(data);

        int pos = 0;

        BaseMessageData<AppDataMessageData> message = new BaseMessageData<>();

        //流水号
        int serialNum = HexUtil.bytesToInt(HexUtil.copyTo(data, pos, 2));
        pos += 2;
        message.setSerialNum(serialNum);

        //版本
        int version = HexUtil.bytesToInt(HexUtil.copyTo(data, pos, 2));
        pos += 2;
        message.setVersion(version);

        //时间
        byte[] dateByte = HexUtil.copyTo(data, pos, 6);
        pos += dateByte.length;
        message.setDate(HexUtil.byteArrayToHex(dateByte));

        //源地址
        byte[] sourceAddressByte = HexUtil.copyTo(data, pos, 6);
        pos += sourceAddressByte.length;
        message.setSourceAddress(HexUtil.byteArrayToHex(sourceAddressByte));

        //目的地址
        byte[] destAddressByte = HexUtil.copyTo(data, pos, 6);
        pos += destAddressByte.length;
        message.setDestAddress(HexUtil.byteArrayToHex(destAddressByte));

        //数据长度
        int dataLen = HexUtil.bytesToInt(HexUtil.copyTo(data, pos, 2));
        pos += 2;

        //命令字
        int cmd = HexUtil.bytesToInt(HexUtil.copyTo(data, pos, 1));
        pos += 1;
        message.setCmd(cmd);

        //check crc
        if (!CheckUtil.verifyCrc(crcByte, calcCrc)) {
            logger.warn("CRC error.FrameCRC = {}. CalcCRC = {}", HexUtil.byteArrayToHex(crcByte), HexUtil.byteArrayToHex(calcCrc));
            return ParserResultHelper.build(ParserStatus.CRC_ERROR, message);
        }
        byte[] appData = new byte[0];
        if (dataLen > 0) {
            //check len
            if (pos + dataLen != data.length) {
                logger.warn("Frame longitudinal disparity,Definition len= {},Frame len ={}", dataLen, (data.length - pos));
                return ParserResultHelper.build(ParserStatus.LONGITUDINAL_DISPARITY, message);
            }
            //数据单元长度太长了.
            if (dataLen > GbtGlobalConfig.MAX_APP_DATA_LEN) {
                logger.warn("AppData too long = {}", dataLen);
                return ParserResultHelper.build(ParserStatus.APP_DATA_TOO_LONG, message);
            }
            //应用数据
            appData = HexUtil.copyTo(data, pos, dataLen);
        }
        return parserByCmd(cmd, appData, message);
    }

    private ParserResult<BaseMessageData<AppDataMessageData>> parserByCmd(int cmd, byte[] appData, BaseMessageData<AppDataMessageData> message) {
        switch (cmd) {
            case CmdConstant.CMD_CONFIRM:
                //收到确认,结束本次通信,只有控制指令成功
                return ParserResultHelper.confirm(message);
            case CmdConstant.CMD_DENY:
                //收到否认
                return ParserResultHelper.deny(message);
            case CmdConstant.CMD_RESPONSE:
                return preParserAppData(message, appData);
            case CmdConstant.CMD_SEND:
                //收到主动上报数据,请求回复数据,解析
                return preParserAppData(message, appData);
            default:
                //未知指令
                break;
        }
        return ParserResultHelper.build(ParserStatus.CMD_ILLEGAL, message);
    }

    /**
     * 转换逻辑
     *
     * @param message
     * @param appData
     * @return
     */
    private ParserResult<BaseMessageData<AppDataMessageData>> preParserAppData(BaseMessageData<AppDataMessageData> message, byte[] appData) {
        //check appData
        if (appData.length < 2) {
            logger.warn("AppData too short. all len =", appData.length);
            return ParserResultHelper.build(ParserStatus.APP_DATA_TOO_SHORT, message);
        }
        AppDataMessageData appDataMessageData;
        try {
            appDataMessageData = this.parserAppData(appData);
        } catch (TypeFlagNotExistException e) {
            logger.warn(e.getMessage(), e);
            e.printStackTrace();
            return ParserResultHelper.build(ParserStatus.PARSER_NOT_FOUND, message);
        } catch (RuntimeException e) {
            logger.warn(e.getMessage(), e);
            e.printStackTrace();
            return ParserResultHelper.build(ParserStatus.UNKNOWN_EXCEPTION, message);
        }
        message.setAppData(appDataMessageData);
        return ParserResultHelper.ok(message);
    }

    /**
     * 转换应用数据
     *
     * @param appData 应用数据
     * @return 转换的目标对象
     * @throws TypeFlagNotExistException 标志类型找不到处理类,返回否认
     * @throws RuntimeException          其他异常,返回否认
     */
    private AppDataMessageData parserAppData(byte[] appData) throws TypeFlagNotExistException {
        int pos = 0;

        byte[] tf = HexUtil.copyTo(appData, pos, 1);
        pos += tf.length;
        //类型标志
        int typeFlag = HexUtil.bytesToInt(tf);

        ITypeFlagParser currentParser = newTypeFlagInstance(typeFlag);

        if (currentParser == null) {
            throw new TypeFlagNotExistException(typeFlag);
        }

        byte[] dc = HexUtil.copyTo(appData, pos, 1);
        pos += dc.length;

        //数据的个数
        int dataInfoCount = HexUtil.bytesToInt(dc);

        byte[][] dataInfo = currentParser.indefiniteLength(pos, dataInfoCount, appData);

        if (dataInfo == null) {
            //定长数据截取
            int singleDataInfoLengthDateTime = currentParser.getSingleDataInfoLength() + currentParser.dateTimeLen();
            dataInfo = new byte[dataInfoCount][];
            int index = 0;

            while (index < dataInfoCount) {
                dataInfo[index++] = HexUtil.copyTo(appData, pos, singleDataInfoLengthDateTime);
                pos += singleDataInfoLengthDateTime;
            }
        } else {
            // 不定长数据截取,已经截取完成,这里为了偏移pos
            for (byte[] temp : dataInfo) {
                pos += temp.length;
            }
        }

        //handler
        List<Map<String, Object>> dataInfoList = new ArrayList<>(dataInfo.length);
        for (byte[] di : dataInfo) {
            Map<String, Object> map = currentParser.parserInfoObjectItem(typeFlag, di);
            dataInfoList.add(map);
        }
        return new AppDataMessageData(typeFlag, dataInfoList);
    }


    private ITypeFlagParser newTypeFlagInstance(int typeFlag) {
        Class<? extends ITypeFlagParser> currentParserClazz = TYPE_FLAG_PARSER.get(typeFlag);
        if (currentParserClazz == null) {
            return null;
        }
        try {
            return currentParserClazz.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            logger.error(e.getMessage(), e);
            e.printStackTrace();
        }
        return null;
    }

}
