package com.vcarecity.telnb.service.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.vcarecity.telnb.constant.NbConstant;
import com.vcarecity.telnb.constant.TelUrlConstant;
import com.vcarecity.telnb.context.IResponseDataContext;
import com.vcarecity.telnb.core.AbstractResponse;
import com.vcarecity.telnb.dto.*;
import com.vcarecity.telnb.properties.TelNbProperties;
import com.vcarecity.telnb.provider.HeaderProvider;
import com.vcarecity.telnb.service.DeviceManagerService;
import com.vcarecity.telnb.util.ResponseHandleUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Kerry on 18/10/22
 */

@Service
@ConditionalOnProperty(value = "tel.nb.enable")
public class DeviceManagerServiceImpl implements DeviceManagerService {

    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;
    private final HeaderProvider headerProvider;
    private final TelNbProperties telNbProperties;

    public DeviceManagerServiceImpl(RestTemplate restTemplate, ObjectMapper objectMapper, HeaderProvider headerProvider, TelNbProperties telNbProperties) {
        this.restTemplate = restTemplate;
        this.objectMapper = objectMapper;
        this.headerProvider = headerProvider;
        this.telNbProperties = telNbProperties;
    }

    @Override
    public AbstractResponse registeredDevice(RegisteredDeviceRequestDTO registeredDevice) throws Exception {
        String value = objectMapper.writeValueAsString(registeredDevice);
        HttpHeaders reqHeader = headerProvider.getReqHeader(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<String> entity = new HttpEntity<>(value, reqHeader);
        Map<String, ?> uri = Collections.singletonMap("appId", telNbProperties.getAppId());
        ResponseEntity<String> response = restTemplate.postForEntity(TelUrlConstant.REGISTER_DEVICE, entity, String.class, uri);
        return ResponseHandleUtil.responseDataHandler(response, RegisteredDeviceResponseDTO.class);
    }

    @Override
    public AbstractResponse updateDeviceInfo(@NotNull String deviceId, UpdateDeviceInfoRequestDTO updateDeviceInfo) throws Exception {
        String value = objectMapper.writeValueAsString(updateDeviceInfo);
        HttpHeaders reqHeader = headerProvider.getAppJsonHeader();
        HttpEntity<String> entity = new HttpEntity<>(value, reqHeader);

        Map<String, String> uris = new HashMap<>(4);
        uris.put(NbConstant.KEY_4_APP_ID, telNbProperties.getAppId());
        uris.put("deviceId", deviceId);

        ResponseEntity<String> response = restTemplate.exchange(TelUrlConstant.UPDATE_DEVICE_INFO, HttpMethod.PUT, entity, String.class, uris);

        return ResponseHandleUtil.responseDataHandler(response, IResponseDataContext.class);
    }

    @Override
    public AbstractResponse deleteDevice(@NotNull String deviceId, @Nullable String cascade) {
        Map<String, String> uris = new HashMap<>(4);
        uris.put(NbConstant.KEY_4_APP_ID, telNbProperties.getAppId());
        uris.put("deviceId", deviceId);
        uris.put("cascade", cascade);

        HttpEntity<String> entity = new HttpEntity<>(headerProvider.getAppJsonHeader());

        ResponseEntity<String> response = restTemplate.exchange(TelUrlConstant.DELETE_DEVICE, HttpMethod.DELETE, entity, String.class, uris);

        return ResponseHandleUtil.responseDataHandler(response, IResponseDataContext.class);
    }

    @Override
    public AbstractResponse queryDeviceActiveStatus(@NotNull String deviceId) {

        Map<String, String> uris = new HashMap<>(4);
        uris.put(NbConstant.KEY_4_APP_ID, telNbProperties.getAppId());
        uris.put("deviceId", deviceId);


        HttpEntity<String> entity = new HttpEntity<>(headerProvider.getAppJsonHeader());

        ResponseEntity<String> response = restTemplate.exchange(TelUrlConstant.QUERY_DEVICE_ACTIVE_STATUS, HttpMethod.GET, entity, String.class, uris);

        return ResponseHandleUtil.responseDataHandler(response, QueryDeviceActiveStatusResponseDTO.class);
    }

    @Override
    public AbstractResponse refreshDeviceKey(String deviceId, RefreshDeviceKeyRequestDTO refreshDeviceKey) throws Exception {
        String value = objectMapper.writeValueAsString(refreshDeviceKey);
        HttpEntity<String> entity = new HttpEntity<>(value, headerProvider.getAppJsonHeader());

        Map<String, String> uris = new HashMap<>(4);
        uris.put(NbConstant.KEY_4_APP_ID, telNbProperties.getAppId());
        uris.put("deviceId", deviceId);

        ResponseEntity<String> response = restTemplate.postForEntity(TelUrlConstant.REGISTER_DEVICE, entity, String.class, uris);

        return ResponseHandleUtil.responseDataHandler(response, RefreshDeviceKeyResponseDTO.class);
    }
}
