package com.vcarecity.telnb.core;

import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static java.util.concurrent.Executors.newSingleThreadScheduledExecutor;

/**
 * @author Kerry on 18/11/07
 * <p>
 * 默认cache
 */
public class DefaultCacheTemplate implements CacheTemplate {


    private static final Logger logger = LoggerFactory.getLogger(DefaultCacheTemplate.class);

    private final Map<String, Object> cacheData;
    private final Map<String, Long> cacheTime;

    @SuppressWarnings("AlibabaThreadPoolCreation")
    public DefaultCacheTemplate() {
        cacheData = new ConcurrentHashMap<>(16);
        cacheTime = new ConcurrentHashMap<>(16);
        //noinspection AlibabaThreadPoolCreation
        ScheduledExecutorService service = newSingleThreadScheduledExecutor();
        Runnable runnable = () -> {
            final Set<String> strings = cacheData.keySet();
            Set<String> expireKey = new HashSet<>();
            strings.forEach(key -> {
                final Long aLong = cacheTime.get(key);
                if (aLong == null) {
                    expireKey.add(key);
                } else if ((System.currentTimeMillis() / 1000) - aLong > 0) {
                    expireKey.add(key);
                }
            });
            expireKey.forEach(cacheData::remove);
        };
        service.scheduleAtFixedRate(runnable, 10, 180, TimeUnit.SECONDS);
    }


    @Nullable
    @Override
    public <T> T get(String key, Class<T> clazz) {
        final Long lastTime = cacheTime.get(key);
        if (lastTime == null) {
            cacheData.remove(key);
            return null;
        }
        if ((System.currentTimeMillis() / 1000) - lastTime > 0) {
            // has expire
            cacheTime.remove(key);
            cacheData.remove(key);
            return null;
        }
        final Object o = cacheData.get(key);
        if (o == null) {
            cacheTime.remove(key);
            return null;
        }
        return (T) o;
    }

    @Override
    public <T> void set(String key, Object value, Class<T> clazz, Long expireIn, TimeUnit timeUnit) {
        cacheData.put(key, value);
        long time = System.currentTimeMillis();
        if (expireIn == null) {
            time += 600L;
        } else {
            time += expireIn;
        }
        cacheTime.put(key, timeUnit.toSeconds(time));
    }

    @Nullable
    @Override
    public Long getExpire(String key, TimeUnit timeUnit) {
        final Object o = cacheData.get(key);
        if (o == null) {
            cacheTime.remove(key);
            return null;
        }
        final Long aLong = cacheTime.get(key);
        if (aLong == null) {
            return null;
        }
        return timeUnit.convert(aLong, TimeUnit.SECONDS);
    }

    @Override
    public void remove(String key) {
        // un support
        cacheData.remove(key);
        cacheTime.remove(key);
    }
}
