package com.vcarecity.telnb.service.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.vcarecity.telnb.constant.TelUrlConstant;
import com.vcarecity.telnb.service.AuthService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * @author KerryZhang
 */

@Service
@ConditionalOnProperty(value = "tel.nb.enable")
public class AuthServiceImpl implements AuthService {

    private static final Logger logger = LoggerFactory.getLogger(AuthServiceImpl.class);

    private final RestTemplate restTemplate;
    private final ObjectMapper objectMapper;


    @Autowired
    public AuthServiceImpl(RestTemplate restTemplate, ObjectMapper objectMapper) {
        this.restTemplate = restTemplate;
        this.objectMapper = objectMapper;
    }

    @Override
    public ResponseEntity<String> requestToken(String appId, String secret) {
        logger.debug("AuthServiceImpl.requestToken. Get Token by appId = {} and secret", appId);

        MultiValueMap<String, String> httpBody = new LinkedMultiValueMap<>();
        httpBody.add("appId", appId);
        httpBody.add("secret", secret);
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        HttpEntity<MultiValueMap<String, String>> request = new HttpEntity<>(httpBody, httpHeaders);
        return restTemplate.postForEntity(TelUrlConstant.APP_AUTH, request, String.class);
    }

    @Override
    public ResponseEntity<String> refreshToken(String appId, String secret, String refreshToken) {

        if (refreshToken == null) {
            //RefreshToken不存在,重新请求
            logger.debug("AuthServiceImpl.refreshToken. refreshToken is null or expire");
            return this.requestToken(appId, secret);
        }

        logger.debug("refreshToken. model = {}. refreshToken = {}", refreshToken);

        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON_UTF8);
        httpHeaders.setAccept(Collections.singletonList(MediaType.APPLICATION_JSON_UTF8));

        Map<String, String> httpBody = new LinkedHashMap<>();
        httpBody.put("appId", appId);
        httpBody.put("secret", secret);
        httpBody.put("refreshToken", refreshToken);

        String body;
        try {
            body = objectMapper.writeValueAsString(httpBody);
        } catch (JsonProcessingException e) {
            e.printStackTrace();
            return null;
        }
        HttpEntity<String> request = new HttpEntity<>(body, httpHeaders);
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(TelUrlConstant.REFRESH_TOKEN, request, String.class);
            if (responseEntity.getStatusCode() == HttpStatus.OK) {
                return responseEntity;
            }
        } catch (RuntimeException re) {
            logger.error(re.getMessage(), re);
            re.printStackTrace();
        }
        return this.requestToken(appId, secret);
    }
}