package com.vcarecity.telnb.service.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.vcarecity.telnb.constant.NbConstant;
import com.vcarecity.telnb.core.CacheTemplate;
import com.vcarecity.telnb.entity.TokenEntity;
import com.vcarecity.telnb.properties.TelNbProperties;
import com.vcarecity.telnb.service.AuthService;
import com.vcarecity.telnb.service.TokenService;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.io.IOException;

/**
 * @author Kerry on 18/11/07
 */

@Service
@ConditionalOnProperty(value = "tel.nb.enable")
public class TokenServiceImpl implements TokenService {

    private static final Logger logger = LoggerFactory.getLogger(TokenServiceImpl.class);

    private final AuthService authService;
    private final CacheTemplate cacheTemplate;
    private final ObjectMapper objectMapper;
    private final TelNbProperties telNbProperties;

    public TokenServiceImpl(AuthService authService,
                            CacheTemplate cacheTemplate,
                            ObjectMapper objectMapper,
                            TelNbProperties telNbProperties) {
        this.authService = authService;
        this.cacheTemplate = cacheTemplate;
        this.objectMapper = objectMapper;
        this.telNbProperties = telNbProperties;
    }


    @Override
    public @Nullable String getAccessToken() {
        final String baseKey = telNbProperties.getKeySpace();
        final String tokenKey = baseKey + NbConstant.KEY_TOKEN_ACCESS_TOKEN;
        String accessToken = cacheTemplate.get(tokenKey);

        if (accessToken != null) {
            return accessToken;
        }
        String refreshTokenKey = baseKey + NbConstant.KEY_TOKEN_REFRESH_TOKEN;
        String refreshToken = cacheTemplate.get(refreshTokenKey);
        ResponseEntity<String> stringResponseEntity = authService.refreshToken(telNbProperties.getAppId(), telNbProperties.getSecret(), refreshToken);
        return accessTokenHandler(stringResponseEntity);
    }

    @Override
    public @Nullable String getRefreshToken() {
        String refreshTokenKey = telNbProperties.getKeySpace() + NbConstant.KEY_TOKEN_REFRESH_TOKEN;
        return cacheTemplate.get(refreshTokenKey);
    }

    @Override
    public String refreshAccessToken() {
        String refreshToken = getRefreshToken();
        if (refreshToken == null) {
            return getAccessToken();
        }
        ResponseEntity<String> stringResponseEntity = authService.refreshToken(telNbProperties.getAppId(), telNbProperties.getSecret(), refreshToken);
        return accessTokenHandler(stringResponseEntity);
    }

    @NotNull
    @Override
    public Long getRefreshTokenExpiresSecond() {
        final Long expire = cacheTemplate.getExpire(telNbProperties.getKeySpace() + NbConstant.KEY_TOKEN_REFRESH_TOKEN);
        if (expire == null) {
            return -1L;
        }
        return expire;
    }

    @NotNull
    @Override
    public Long getAccessTokenExpiresSecond() {
        final Long expire = cacheTemplate.getExpire(telNbProperties.getKeySpace() + NbConstant.KEY_TOKEN_ACCESS_TOKEN);
        if (expire == null) {
            return -1L;
        }
        return expire;
    }

    @Override
    public Long getTokenLastUpdateTime() {
        final String s = cacheTemplate.get(telNbProperties.getKeySpace() + NbConstant.KEY_TOKEN_LAST_TIME);
        if (s != null) {
            return Long.parseLong(s);
        }
        return 0L;
    }

    @Override
    public void updateTokenLastTime() {
        String key = telNbProperties.getKeySpace() + NbConstant.KEY_TOKEN_LAST_TIME;
        cacheTemplate.set(key, String.valueOf(System.currentTimeMillis()), telNbProperties.getRefreshTokenExpire().longValue());
    }

    @Override
    public String getScope() {
        return cacheTemplate.get(telNbProperties.getKeySpace() + NbConstant.KEY_TOKEN_SCOPE);
    }

    @Override
    public String getTokenType() {
        return cacheTemplate.get(telNbProperties.getKeySpace() + NbConstant.KEY_TOKEN_TOKEN_TYPE);
    }


    private String accessTokenHandler(ResponseEntity<String> stringResponseEntity) {
        String baseKey = telNbProperties.getKeySpace();
        if (stringResponseEntity == null) {
            return null;
        } else if (stringResponseEntity.getStatusCode() == HttpStatus.OK) {
            TokenEntity tokenEntity;
            try {
                tokenEntity = objectMapper.readValue(stringResponseEntity.getBody(), TokenEntity.class);
            } catch (IOException e) {
                logger.error(e.getMessage(), e);
                e.printStackTrace();
                return null;
            }

            logger.debug("request access token = {}", tokenEntity.getAccessToken());

            int expiresIn = tokenEntity.getExpiresIn();

            cacheTemplate.set(baseKey + NbConstant.KEY_TOKEN_ACCESS_TOKEN, tokenEntity.getAccessToken(), (long) expiresIn);
            cacheTemplate.set(baseKey + NbConstant.KEY_TOKEN_REFRESH_TOKEN, tokenEntity.getRefreshToken(), Long.valueOf(telNbProperties.getRefreshTokenExpire()));
            cacheTemplate.set(baseKey + NbConstant.KEY_TOKEN_SCOPE, tokenEntity.getScope(), (long) expiresIn);
            cacheTemplate.set(baseKey + NbConstant.KEY_TOKEN_TOKEN_TYPE, tokenEntity.getTokenType(), (long) expiresIn);
            cacheTemplate.set(baseKey + NbConstant.KEY_TOKEN_LAST_TIME, String.valueOf(System.currentTimeMillis()), Long.valueOf(telNbProperties.getRefreshTokenExpire()));
            return tokenEntity.getAccessToken();
        } else if (stringResponseEntity.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            logger.error("401.UNAUTHORIZED. {}", stringResponseEntity.getBody());
        }
        return null;
    }
}

