package com.vcarecity.dtu.service.impl;

import com.vcarecity.allcommon.exception.NoRequireKeyException;
import com.vcarecity.allcommon.util.HexUtil;
import com.vcarecity.common.context.BaseJsonViewBean;
import com.vcarecity.common.service.IConvertJsonService;
import com.vcarecity.common.service.sub.IConverter;
import com.vcarecity.common.util.CrcUtil;
import com.vcarecity.dtu.issued.read.IssuedRead;
import com.vcarecity.dtu.issued.read.IssuedReadCommon;
import com.vcarecity.dtu.issued.write.IssuedWrite;
import com.vcarecity.dtu.response.BatteryLoginResponse;
import com.vcarecity.dtu.response.HeartBeatResponse;
import com.vcarecity.dtu.response.LoginResponse;
import com.vcarecity.dtu.response.ReportResponse;
import com.vcarecity.dtucommon.constant.DtuProtocolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author Kerry on 18/06/27
 */

public class ConvertJsonServiceImpl implements IConvertJsonService<BaseJsonViewBean> {

    private static Logger logger = LoggerFactory.getLogger(ConvertJsonServiceImpl.class);

    protected static ConcurrentHashMap<String, IConverter> CONVERTER_CACHE = new ConcurrentHashMap<>(64);

    private static final String PATH_ISSUED_READ = IssuedRead.class.getName();
    private static final String PATH_ISSUED_WRITE = IssuedWrite.class.getName();

    @SuppressWarnings("Duplicates")
    @Override
    public byte[] convertJson(BaseJsonViewBean messageContext) throws NoRequireKeyException {
        if (messageContext == null) {
            return null;
        }

        IConverter converter;
        int funCode = messageContext.getFunctionWord();
        byte[] body = new byte[0];

        switch (messageContext.getFunctionWord()) {
            case DtuProtocolConfig.CODE_HEART_BEAT:
                funCode -= 128;
                messageContext.setSn(null);
                messageContext.setEquipmentNo(null);
                converter = new HeartBeatResponse();
                body = converter.convert(messageContext);
                break;
            case DtuProtocolConfig.CODE_LOGIN:
                funCode -= 128;
                messageContext.setSn(null);
                messageContext.setEquipmentNo(null);
                converter = new LoginResponse();
                body = converter.convert(messageContext);
                break;
            case DtuProtocolConfig.CODE_BATTERY_LOGIN:
                funCode -= 128;

                messageContext.setEquipmentNo(null);
                converter = new BatteryLoginResponse();
                body = converter.convert(messageContext);
                // 这里必须放到 convert() 后面
                messageContext.setSn(null);
                break;
            case DtuProtocolConfig.CODE_REPORT:
                funCode -= 128;
                converter = getReportResponse(messageContext);
                body = converter.convert(messageContext);
                break;
            case DtuProtocolConfig.CODE_READ:
                List<BaseJsonViewBean.DataItemDTO> dataItems = messageContext.getDataItems();
                byte[] eqAndSnRead = null;
                for (BaseJsonViewBean.DataItemDTO dataItem : dataItems) {
                    converter = getIssuedReadConverter(dataItem.getId());
                    if (eqAndSnRead == null) {
                        eqAndSnRead = converter.customSnAndEquipmentNo(messageContext.getSn(), messageContext.getEquipmentNo());
                        if (eqAndSnRead != null) {
                            body = HexUtil.concatAll(eqAndSnRead, body);
                        }
                    }
                    if (dataItem.getId() == null) {
                        continue;
                    }
                    byte[] idArrayRead = HexUtil.intToByteArray(dataItem.getId(), DtuProtocolConfig.DTU_INFO_ID_LEN);
                    body = HexUtil.concatAll(body, idArrayRead);
                    //concat message...
                    byte[] onceArrayData = converter.convertItemBody(dataItem.getId(), dataItem.getData());
                    if (onceArrayData != null) {
                        body = HexUtil.concatAll(body, onceArrayData);
                    }
                    //byte[] bytes = converter.itemBodyConverter(messageContext.getSn(), messageContext.getEquipmentNo(), dataItem.getId(), dataItem.getData());
                    //if (bytes != null) {
                    //    body = HexUtil.concatAll(body, bytes);
                    //}
                }
                break;
            case DtuProtocolConfig.CODE_WRITE:
                List<BaseJsonViewBean.DataItemDTO> dataItemsWrite = messageContext.getDataItems();
                byte[] eqAndSnWrite = null;
                for (BaseJsonViewBean.DataItemDTO dataItem : dataItemsWrite) {
                    converter = getIssuedWriteConverter(dataItem.getId());
                    if (converter == null) {
                        continue;
                    }
                    if (eqAndSnWrite == null) {
                        eqAndSnWrite = converter.customSnAndEquipmentNo(messageContext.getSn(), messageContext.getEquipmentNo());
                        if (eqAndSnWrite != null) {
                            body = HexUtil.concatAll(eqAndSnWrite, body);
                        }
                    }
                    if (dataItem.getId() == null) {
                        continue;
                    }
                    byte[] idArrayWrite = HexUtil.intToByteArray(dataItem.getId(), DtuProtocolConfig.DTU_INFO_ID_LEN);
                    body = HexUtil.concatAll(body, idArrayWrite);
                    byte[] onceArrayData = converter.convertItemBody(dataItem.getId(), dataItem.getData());
                    if (onceArrayData != null) {
                        body = HexUtil.concatAll(body, onceArrayData);
                    }
                    //byte[] bytes = converter.itemBodyConverter(messageContext.getSn(), messageContext.getEquipmentNo(), dataItem.getId(), dataItem.getData());
                    //if (bytes != null) {
                    //    body = HexUtil.concatAll(body, bytes);
                    //}

                }
                break;
            default:
                return null;
        }
        messageContext.setFunctionWord(funCode);
        //添加消息头+消息体
        byte[] msgBody = HexUtil.concatAll(
                HexUtil.hexToByteArray(messageContext.getUnitNo(), DtuProtocolConfig.DTU_DEVICE_ID_LEN),
                HexUtil.intToByteArray(messageContext.getFunctionWord(), DtuProtocolConfig.DTU_FUNC_LEN),
                HexUtil.intToByteArray(body.length, DtuProtocolConfig.DTU_LEN_LEN),
                body);

        //CRC
        int crcCode = CrcUtil.validate(msgBody);
        //添加头尾..
        return HexUtil.concatAll(DtuProtocolConfig.DTU_MASK,
                msgBody,
                HexUtil.intToByteArray(crcCode, DtuProtocolConfig.DTU_CRC_LEN));

    }

    private IConverter getIssuedReadConverter(Integer id) {
        try {
            return this.newConverterInstance(PATH_ISSUED_READ + String.valueOf(id));
        } catch (IllegalAccessException | InstantiationException | ClassNotFoundException e) {
            logger.warn("can not getIssuedReadConverter class on id = {}", id);
        }
        return new IssuedReadCommon();
    }

    private IConverter getIssuedWriteConverter(Integer id) {
        try {
            return this.newConverterInstance(PATH_ISSUED_WRITE + String.valueOf(id));
        } catch (ClassNotFoundException | IllegalAccessException | InstantiationException e) {
            logger.warn("can not getIssuedWriteConverter class on id = {}", id);
        }
        return null;
    }

    private IConverter getReportResponse(BaseJsonViewBean messageContext) {
        //update here...
        return new ReportResponse();
    }

    private IConverter newConverterInstance(String packageName) throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        IConverter converter = CONVERTER_CACHE.get(packageName);
        if (converter != null) {
            return converter;
        }
        Class<?> forName = Class.forName(packageName);
        converter = (IConverter) forName.newInstance();
        CONVERTER_CACHE.put(packageName, converter);
        return converter;
    }

    /**
     * may be other 实现
     *
     * @param converterCache
     */
    public static void setConverterCache(ConcurrentHashMap<String, IConverter> converterCache) {
        CONVERTER_CACHE = converterCache;
    }
}
