/*
 * Decompiled with CFR 0.152.
 */
package me.prettyprint.cassandra.service;

import java.io.IOException;
import java.util.List;
import me.prettyprint.cassandra.service.CassandraClient;
import me.prettyprint.cassandra.service.CassandraClientMonitor;
import me.prettyprint.cassandra.service.CassandraClientPool;
import me.prettyprint.cassandra.service.Keyspace;
import me.prettyprint.cassandra.service.Operation;
import me.prettyprint.cassandra.service.PoolExhaustedException;
import me.prettyprint.cassandra.service.SkipHostException;
import org.apache.cassandra.thrift.InvalidRequestException;
import org.apache.cassandra.thrift.TimedOutException;
import org.apache.cassandra.thrift.UnavailableException;
import org.apache.thrift.TException;
import org.apache.thrift.transport.TTransportException;
import org.perf4j.StopWatch;
import org.perf4j.slf4j.Slf4JStopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class FailoverOperator {
    private static final Logger log = LoggerFactory.getLogger(FailoverOperator.class);
    private final CassandraClient.FailoverPolicy failoverPolicy;
    private final List<String> knownHosts;
    private final CassandraClientMonitor monitor;
    private CassandraClient client;
    private final CassandraClientPool clientPools;
    private final Keyspace keyspace;

    public FailoverOperator(CassandraClient.FailoverPolicy policy, List<String> hosts, CassandraClientMonitor monitor, CassandraClient client, CassandraClientPool clientPools, Keyspace keyspace) {
        this.failoverPolicy = policy;
        this.knownHosts = hosts;
        this.monitor = monitor;
        this.client = client;
        this.clientPools = clientPools;
        this.keyspace = keyspace;
    }

    public void operate(Operation<?> op) throws InvalidRequestException, UnavailableException, TException, TimedOutException {
        Slf4JStopWatch stopWatch = new Slf4JStopWatch();
        int retries = Math.min(this.failoverPolicy.getNumRetries() + 1, this.knownHosts.size());
        boolean isFirst = true;
        try {
            while (retries > 0) {
                if (!isFirst) {
                    --retries;
                }
                try {
                    boolean success = this.operateSingleIteration(op, (StopWatch)stopWatch, retries, isFirst);
                    if (success) {
                        return;
                    }
                }
                catch (SkipHostException e) {
                    log.warn("Skip-host failed ", (Throwable)e);
                }
                isFirst = false;
            }
        }
        catch (InvalidRequestException e) {
            this.monitor.incCounter(op.failCounter);
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            throw e;
        }
        catch (UnavailableException e) {
            this.invalidate();
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            this.monitor.incCounter(op.failCounter);
            throw e;
        }
        catch (TException e) {
            this.invalidate();
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            this.monitor.incCounter(op.failCounter);
            throw e;
        }
        catch (TimedOutException e) {
            this.invalidate();
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            this.monitor.incCounter(op.failCounter);
            throw e;
        }
        catch (PoolExhaustedException e) {
            log.warn("Pool is exhausted", (Throwable)e);
            this.monitor.incCounter(op.failCounter);
            this.monitor.incCounter(CassandraClientMonitor.Counter.POOL_EXHAUSTED);
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            throw new UnavailableException();
        }
        catch (IllegalStateException e) {
            log.error("Client Pool is already closed, cannot obtain new clients.", (Throwable)e);
            this.monitor.incCounter(op.failCounter);
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            throw new UnavailableException();
        }
        catch (IOException e) {
            this.invalidate();
            this.monitor.incCounter(op.failCounter);
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            throw new UnavailableException();
        }
        catch (Exception e) {
            log.error("Cannot retry failover, got an Exception", (Throwable)e);
            this.monitor.incCounter(op.failCounter);
            stopWatch.stop(op.stopWatchTagName + ".fail_");
            throw new UnavailableException();
        }
    }

    private boolean operateSingleIteration(Operation<?> op, StopWatch stopWatch, int retries, boolean isFirst) throws InvalidRequestException, TException, TimedOutException, PoolExhaustedException, Exception, UnavailableException, TTransportException {
        log.debug("Performing operation on {}; retries: {}", (Object)this.client.getUrl(), (Object)retries);
        try {
            op.executeAndSetResult(this.client.getCassandra());
            log.debug("Operation succeeded on {}", (Object)this.client.getUrl());
            stopWatch.stop(op.stopWatchTagName + ".success_");
            return true;
        }
        catch (TimedOutException e) {
            log.warn("Got a TimedOutException from {}. Num of retries: {}", (Object)this.client.getUrl(), (Object)retries);
            if (retries == 0) {
                throw e;
            }
            this.skipToNextHost(isFirst, false);
            this.monitor.incCounter(CassandraClientMonitor.Counter.RECOVERABLE_TIMED_OUT_EXCEPTIONS);
        }
        catch (UnavailableException e) {
            log.warn("Got a UnavailableException from {}. Num of retries: {}", (Object)this.client.getUrl(), (Object)retries);
            if (retries == 0) {
                throw e;
            }
            this.skipToNextHost(isFirst, true);
            this.monitor.incCounter(CassandraClientMonitor.Counter.RECOVERABLE_UNAVAILABLE_EXCEPTIONS);
        }
        catch (TTransportException e) {
            log.warn("Got a TTransportException from {}. Num of retries: {}", (Object)this.client.getUrl(), (Object)retries);
            if (retries == 0) {
                throw e;
            }
            this.skipToNextHost(isFirst, true);
            this.monitor.incCounter(CassandraClientMonitor.Counter.RECOVERABLE_TRANSPORT_EXCEPTIONS);
        }
        return false;
    }

    private void skipToNextHost(boolean isRetrySameHostAgain, boolean invalidateAllConnectionsToCurrentHost) throws SkipHostException {
        String nextHost;
        log.info("Skipping to next host. Current host is: {}", (Object)this.client.getUrl());
        this.invalidate();
        if (invalidateAllConnectionsToCurrentHost) {
            this.clientPools.invalidateAllConnectionsToHost(this.client);
        }
        String string = nextHost = isRetrySameHostAgain ? this.client.getUrl() : this.getNextHost(this.client.getUrl(), this.client.getIp());
        if (nextHost == null) {
            log.error("Unable to find next host to skip to at {}", (Object)this.toString());
            throw new SkipHostException("Unable to failover to next host");
        }
        try {
            this.client = this.clientPools.borrowClient(nextHost, this.client.getPort());
        }
        catch (IllegalStateException e) {
            throw new SkipHostException(e);
        }
        catch (PoolExhaustedException e) {
            throw new SkipHostException(e);
        }
        catch (Exception e) {
            throw new SkipHostException(e);
        }
        this.monitor.incCounter(CassandraClientMonitor.Counter.SKIP_HOST_SUCCESS);
        log.info("Skipped host. New host is: {}", (Object)this.client.getUrl());
    }

    private void invalidate() {
        try {
            this.clientPools.invalidateClient(this.client);
            this.client.removeKeyspace(this.keyspace);
        }
        catch (Exception e) {
            log.error("Unable to invalidate client {}. Will continue anyhow.", (Object)this.client);
        }
    }

    private String getNextHost(String url, String ip) {
        int size = this.knownHosts.size();
        if (size < 1) {
            return null;
        }
        for (int i = 0; i < this.knownHosts.size(); ++i) {
            if (!url.equals(this.knownHosts.get(i)) && !ip.equals(this.knownHosts.get(i))) continue;
            return this.knownHosts.get((i + 1) % size);
        }
        log.error("The URL {} wasn't found in the knownHosts", (Object)url);
        return null;
    }
}

